package org.etsi.mts.tdl.yang2tdl;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

import org.etsi.mts.tdl.CollectionDataType;
import org.etsi.mts.tdl.DataType;
import org.etsi.mts.tdl.EnumDataType;
import org.etsi.mts.tdl.Extension;
import org.etsi.mts.tdl.Member;
import org.etsi.mts.tdl.SimpleDataInstance;
import org.etsi.mts.tdl.SimpleDataType;
import org.etsi.mts.tdl.StructuredDataInstance;
import org.etsi.mts.tdl.StructuredDataType;
import org.etsi.mts.tdl.tdlFactory;
import org.etsi.mts.tdl.tdlPackage;
import org.etsi.mts.tdl.transform.AbstractTranslator;
import org.opendaylight.yangtools.odlext.parser.MountStatementSupport;
import org.opendaylight.yangtools.yang.common.QName;
import org.opendaylight.yangtools.yang.model.api.CaseSchemaNode;
import org.opendaylight.yangtools.yang.model.api.ContainerLike;
import org.opendaylight.yangtools.yang.model.api.ContainerSchemaNode;
import org.opendaylight.yangtools.yang.model.api.DataNodeContainer;
import org.opendaylight.yangtools.yang.model.api.DataSchemaNode;
import org.opendaylight.yangtools.yang.model.api.LeafListSchemaNode;
import org.opendaylight.yangtools.yang.model.api.LeafSchemaNode;
import org.opendaylight.yangtools.yang.model.api.ListSchemaNode;
import org.opendaylight.yangtools.yang.model.api.SchemaContext;
import org.opendaylight.yangtools.yang.model.api.TypeDefinition;
import org.opendaylight.yangtools.yang.model.api.TypedDataSchemaNode;
import org.opendaylight.yangtools.yang.model.api.stmt.LeafStatement;
import org.opendaylight.yangtools.yang.model.api.type.EnumTypeDefinition;
import org.opendaylight.yangtools.yang.model.api.type.EnumTypeDefinition.EnumPair;
import org.opendaylight.yangtools.yang.model.api.type.IdentityrefTypeDefinition;
import org.opendaylight.yangtools.yang.model.api.type.LeafrefTypeDefinition;
import org.opendaylight.yangtools.yang.model.repo.api.YangTextSchemaSource;
import org.opendaylight.yangtools.yang.model.ri.type.BaseTypes;
import org.opendaylight.yangtools.yang.model.ri.stmt.impl.eff.DeclaredCaseEffectiveStatement;
import org.opendaylight.yangtools.yang.model.ri.stmt.impl.eff.EmptyChoiceEffectiveStatement;
import org.opendaylight.yangtools.yang.model.ri.stmt.impl.eff.EmptyLeafEffectiveStatement;
import org.opendaylight.yangtools.yang.model.ri.stmt.impl.eff.EmptyLeafListEffectiveStatement;
import org.opendaylight.yangtools.yang.model.ri.stmt.impl.eff.RegularChoiceEffectiveStatement;
import org.opendaylight.yangtools.yang.model.ri.stmt.impl.eff.RegularLeafEffectiveStatement;
import org.opendaylight.yangtools.yang.model.ri.stmt.impl.eff.RegularLeafListEffectiveStatement;
import org.opendaylight.yangtools.yang.model.ri.stmt.impl.eff.SlimLeafListEffectiveStatement;
import org.opendaylight.yangtools.yang.model.ri.stmt.impl.eff.UndeclaredCaseEffectiveStatement;
import org.opendaylight.yangtools.yang.model.spi.meta.AbstractDeclaredEffectiveStatement.Default;
import org.opendaylight.yangtools.yang.parser.api.YangParserConfiguration;
import org.opendaylight.yangtools.yang.parser.api.YangSyntaxErrorException;
import org.opendaylight.yangtools.yang.parser.rfc7950.reactor.RFC7950Reactors;
import org.opendaylight.yangtools.yang.parser.rfc7950.repo.YangStatementStreamSource;
import org.opendaylight.yangtools.yang.parser.spi.meta.ModelProcessingPhase;
import org.opendaylight.yangtools.yang.parser.spi.meta.ReactorException;
import org.opendaylight.yangtools.yang.parser.stmt.reactor.CrossSourceStatementReactor;
import org.opendaylight.yangtools.yang.parser.stmt.reactor.CrossSourceStatementReactor.BuildAction;

public class Yang2TDLTranslator extends AbstractTranslator {

    private static final String REFERENCE_DESCRIPTION = "ReferenceDescription";
	private static final String LIST_KEY = "ListKey";
	private static final String LEAF_REF_PATH = "LeafRefPath";

	//inherited entry point
	@Override
	public void translate(String targetFilename) throws Exception {
		String sourceMappingTag = "SOURCE_MAPPING";
		drm = getTypeFor(sourceMappingTag, tdlPackage.Literals.DATA_RESOURCE_MAPPING);
		drm.setResourceURI(new File(targetFilename).getName());
		
		//TODO: resolve files
		//TODO: extract
        CrossSourceStatementReactor reactor = createReactor();
		BuildAction build = reactor.newBuild();
		Files.list(Path.of("samples", "ietf"))
			.filter(p->p.toString().endsWith("yang"))
			.forEach(p -> 
			build.addLibSource(moduleFromPath(p.toString()))
		);
		Files.list(Path.of("samples", "nfv"))
			.filter(p->p.toString().endsWith("yang"))
			.forEach(p -> 
			build.addSource(moduleFromPath(p.toString()))
		);
//		Files.list(Path.of("samples", "mwt"))
//			.filter(p->p.toString().endsWith("yang"))
//			.forEach(p -> 
//			build.addSource(moduleFromPath(p.toString()))
//		);
//		Files.list(Path.of("samples", "simple"))
//			.filter(p->p.toString().endsWith("yang"))
//			.forEach(p -> 
//			build.addSource(moduleFromPath(p.toString()))
//		);
//		build.addSource(moduleFromPath(targetFilename));
		SchemaContext schemaContext = build.buildEffective();
		//TODO: handle multimodule paths?
		//TODO: include meta-data
		//TODO: library of built-in types
		//TODO: add extensions
		translate(schemaContext);
	}


	private void translate(SchemaContext schemaContext) {
		for (DataSchemaNode c : schemaContext.getChildNodes()) {
			translate(c);
		}
	}

	private DataType translate(DataSchemaNode c) {
		return translate(c, null);
	}
	
	private DataType translate(DataSchemaNode c, String parent) {
		//wrapper for meta-data
		//TODO: make optional
		//TODO: resolve issues with annotations
		//TODO: use notes?
		DataType dataType = translateOnly(c, parent);
//		c.getDescription().ifPresent(e-> {
//			annotateWith(dataType, REFERENCE_DESCRIPTION, e);
//		});
		return dataType;
	}
	
	private DataType translateOnly(DataSchemaNode c, String parent) {
		String baseName = cleanName(c.getQName().getLocalName());
		String name = baseName;
		if (parent != null) {
			name = parent+"___"+name;
		}
		if (c instanceof ContainerSchemaNode) {
			//DONE: remove wrapper container structured data type of only a collection is contained
			Collection<? extends DataSchemaNode> childNodes = ((ContainerSchemaNode) c).getChildNodes();
			DataSchemaNode first = childNodes.iterator().next();
			if (childNodes.size() == 1 && first instanceof ListSchemaNode) {
				//create collection
				return translate(first, parent); //parent?
			} else {
				//create structured data type
				StructuredDataType dataType = translateStructuredDataType((ContainerSchemaNode) c, baseName, name);
				return dataType;
			}
		} else if (c instanceof ListSchemaNode) {
			StructuredDataType itemType = translateStructuredDataType((ListSchemaNode) c, baseName, name);
			CollectionDataType collectionType = getCollectionDataTypeFor(itemType);

			//TODO: add key annotation?
			List<QName> keyDefinition = ((ListSchemaNode) c).getKeyDefinition();
			String key = keyDefinition.stream()
					.map(QName::getLocalName)
					.collect(Collectors.joining(", "));
			// TODO: make optional
//			annotateWith(collectionType, LIST_KEY, key);
			return collectionType;
		} else if (c instanceof RegularLeafListEffectiveStatement) {
			//TODO: what is the difference?
			TypeDefinition<?> type = ((RegularLeafListEffectiveStatement) c).getType();
			CollectionDataType collectionType = translateSimpleDataCollection(name, type);
			return collectionType;
		} else if (c instanceof EmptyLeafListEffectiveStatement) {
			//DONE: unify and reuse
			TypeDefinition<?> type = ((EmptyLeafListEffectiveStatement) c).getType();
			CollectionDataType collectionType = translateSimpleDataCollection(name, type);
			return collectionType;
		} else if (c instanceof SlimLeafListEffectiveStatement) {
			//TODO: check and remove?
			TypeDefinition<?> type = ((SlimLeafListEffectiveStatement) c).getType();
			CollectionDataType collectionType = translateSimpleDataCollection(name, type);
			return collectionType;
		} else if (c instanceof EmptyLeafEffectiveStatement) {
			TypeDefinition<?> type = ((EmptyLeafEffectiveStatement) c).getType();
			SimpleDataType dataType = translateSimpleDataType(name, type);
			return dataType;
		} else if (c instanceof RegularLeafEffectiveStatement) {
			//TODO: what is the difference? is it used
			String typeName = ((RegularLeafEffectiveStatement) c).getType().getQName().getLocalName();
			SimpleDataType dataType = getSimpleDataTypeFor(typeName);
			return dataType;
		} else if (c instanceof RegularChoiceEffectiveStatement) {
			//TODO: difference?
			StructuredDataType dataType = getStructuredDataTypeFor(name);
			for (CaseSchemaNode child : ((RegularChoiceEffectiveStatement) c).getCases()) {
				translateMember(baseName, dataType, child);
			}
			return dataType;
		} else if (c instanceof EmptyChoiceEffectiveStatement) {
			//TODO: add annotation choice
			//TODO: need to skip cases?
			//TODO: structured or simple?!
			StructuredDataType dataType = getStructuredDataTypeFor(name);
			for (CaseSchemaNode child : ((EmptyChoiceEffectiveStatement) c).getCases()) {
				translateMember(baseName, dataType, child);
			}
			return dataType;
		} else if (c instanceof UndeclaredCaseEffectiveStatement) {
			//choice -> container? (no case)
			return translateStructuredDataType((DataNodeContainer) c, baseName, name);
		} else if (c instanceof DeclaredCaseEffectiveStatement) {
			//choice -> case
			//TODO: compare and check
			return translateStructuredDataType((DataNodeContainer) c, baseName, name);
		} else {
			System.out.println("TODO:: "+c);
			SimpleDataType todoType = getSimpleDataTypeFor("TODO");
			return todoType;
		}
	}

	private CollectionDataType translateSimpleDataCollection(String name, TypeDefinition<?> type) {
		SimpleDataType itemType = translateSimpleDataType(name, type);
		CollectionDataType collectionType = getCollectionDataTypeFor(itemType);
		return collectionType;
	}

	private SimpleDataType translateSimpleDataType(String name, TypeDefinition<?> type) {
		if (type instanceof IdentityrefTypeDefinition) {
			//TODO: handle further?
//			System.out.println("identityref:"+name+" -> "+((IdentityrefTypeDefinition)type).getIdentities());
		}
		if (type instanceof LeafrefTypeDefinition) {
			//TODO: handle further?
//			System.out.println("leafref:"+name+" -> "+((LeafrefTypeDefinition)type).getPathStatement().getOriginalString());
		}

		TypeDefinition<?> baseType = type.getBaseType();
		String typeName = type.getQName().getLocalName();
//		typeName = name; //DONE: fine? -> seems to be -> causes lots of redundant types
		if (baseType instanceof EnumTypeDefinition) { 
			//DONE: handle enumeration -> add instances rather than extension
			EnumDataType dataType = getEnumDataTypeFor(typeName);
			for (EnumPair e : ((EnumTypeDefinition)baseType).getValues()) {
				SimpleDataInstance eInstance = getSimpleDataInstanceFor(e.getName());
				eInstance.setDataType(dataType);
				dataType.getValue().add(eInstance);
			}
			//TODO: add default?
			return dataType;
		} else {
			//TODO: add constraints?
			SimpleDataType dataType = getSimpleDataTypeFor(typeName);
			//TODO: what if extension is already present?
			if (baseType != null) {
				//add extension
				SimpleDataType superType = getSimpleDataTypeFor(baseType.getQName().getLocalName());
				Extension e = tdlFactory.eINSTANCE.createExtension();
				e.setExtending(superType);
				dataType.setExtension(e);
			}
			return dataType;
		}
	}


	private void translateMember(String baseName, StructuredDataType dataType, DataSchemaNode child) {
		DataType memberType = translate(child, baseName);
		String propertyName = child.getQName().getLocalName();
		Member m = getContentWithName((String) propertyName, dataType, tdlPackage.Literals.MEMBER);
		m.setDataType(memberType);
		if (m.container() == null) {
			dataType.getMember().add(m);
		}
		
		if (child instanceof TypedDataSchemaNode) {
			TypeDefinition<?> type = ((TypedDataSchemaNode) child).getType();
			if (type instanceof LeafrefTypeDefinition) {
				String value = ((LeafrefTypeDefinition)type).getPathStatement().getOriginalString();
				annotateWith(m, LEAF_REF_PATH, value);
			}
		}
	}


	private StructuredDataType translateStructuredDataType(DataNodeContainer c, String baseName, String name) {
		StructuredDataType dataType = getStructuredDataTypeFor(name);
		for (DataSchemaNode child : c.getChildNodes()) {
			translateMember(baseName, dataType, child);
		}
		return dataType;
	}
	
	//----------------------------------------------------------
	//TODO: clean up below
	
	
	public static void main(String[] args) throws Exception {
        CrossSourceStatementReactor reactor = createReactor();

//        processStatic(reactor);

		BuildAction build = reactor.newBuild();
		Files.list(Path.of("samples", "ietf")).forEach(p -> 
			build.addSource(moduleFromPath(p.toString()))
		);
//		Files.list(Path.of("samples", "nfv")).forEach(p -> 
//			build.addSource(moduleFromPath(p.toString()))
//		);
		Files.list(Path.of("samples", "simple")).forEach(p -> 
			build.addSource(moduleFromPath(p.toString()))
		);
		
//		build.addSource(moduleFromPath("samples/nfv/etsi-nfv-descriptors.yang"));
		
		SchemaContext schemaContext = build.buildEffective();
//		
//		Set<Module> modules = schemaContext.getModules();
//		Set<DataSchemaNode> dataSchemaNodes = schemaContext.getDataDefinitions();
//		
//		dataSchemaNodes.forEach(System.out::println);
//		schemaContext.getChildNodes().forEach(System.out::println);
////		schemaContext.getTypeDefinitions().forEach(System.out::println);
		expand("  ", schemaContext);
	}


	private static CrossSourceStatementReactor createReactor() {
		CrossSourceStatementReactor reactor = RFC7950Reactors.vanillaReactorBuilder()
        .addStatementSupport(ModelProcessingPhase.FULL_DECLARATION,
            new MountStatementSupport(YangParserConfiguration.DEFAULT))
        .build();
		return reactor;
	}

	private static void processStatic(CrossSourceStatementReactor reactor) throws ReactorException {
		reactor.newBuild().addSources(
//                moduleFromResources("/ietf/ietf-yang-types@2013-07-15.yang"),
//                moduleFromResources("/ietf/ietf-restconf@2017-01-26.yang"),
//                moduleFromResources("/ietf/ietf-inet-types@2013-07-15.yang"),
//                moduleFromResources("/nfv/etsi-nfv-common.yang"),
//                moduleFromResources("/nfv/etsi-nfv-nsd.yang"),
//                moduleFromResources("/nfv/etsi-nfv-pnfd.yang"),
//                moduleFromResources("/nfv/etsi-nfv-vnfd.yang"),
//                moduleFromResources("/nfv/etsi-nfv-descriptors.yang"),
//                moduleFromResources("/nfv/etsi-nfv-ns.yang"),
//                moduleFromResources("/nfv/etsi-nfv-pnf.yang"), 
//                moduleFromResources("/nfv/etsi-nfv-vnf.yang")
                moduleFromPath("samples/ietf/ietf-yang-types@2013-07-15.yang"),
                moduleFromPath("samples/ietf/ietf-restconf@2017-01-26.yang"),
                moduleFromPath("samples/ietf/ietf-inet-types@2013-07-15.yang"),
                moduleFromPath("samples/nfv/etsi-nfv-common.yang"),
                moduleFromPath("samples/nfv/etsi-nfv-nsd.yang"),
                moduleFromPath("samples/nfv/etsi-nfv-pnfd.yang"),
                moduleFromPath("samples/nfv/etsi-nfv-vnfd.yang"),
                moduleFromPath("samples/nfv/etsi-nfv-descriptors.yang"),
                moduleFromPath("samples/nfv/etsi-nfv-ns.yang"),
                moduleFromPath("samples/nfv/etsi-nfv-pnf.yang"), 
                moduleFromPath("samples/nfv/etsi-nfv-vnf.yang")
            )
                // moduleFromResources("/simple/sports.yang"))
            .buildEffective()
            .getChildNodes().forEach(e->System.out.println(e.getQName()));
            ;
	}
	
    private static YangStatementStreamSource moduleFromResources(final String resourceName) {
        try {
            return YangStatementStreamSource.create(YangTextSchemaSource.forResource(resourceName));
        } catch (final YangSyntaxErrorException | IOException e) {
            throw new IllegalStateException("Failed to find resource " + resourceName, e);
        }
    }
    private static YangStatementStreamSource moduleFromPath(String path) {
    	return moduleFromPath(Path.of(path));
    }
    private static YangStatementStreamSource moduleFromPath(Path path) {
        try {
            return YangStatementStreamSource.create(YangTextSchemaSource.forPath(path));
        } catch (final YangSyntaxErrorException | IOException e) {
            throw new IllegalStateException("Failed to find resource " + path.toString(), e);
        }
    }
	
	public static void expand(String prefix, ContainerLike node) {
		for (DataSchemaNode c : node.getChildNodes()) {
			expand(prefix, c);
		}
	}

	public static void expand(String prefix, ListSchemaNode node) {
		for (DataSchemaNode c : node.getChildNodes()) {
			expand(prefix, c);
		}
	}

	private static void expand(String prefix, DataSchemaNode c) {
		//TODO: this is not accurately reflecting the structures..
		if (c instanceof LeafListSchemaNode) {
			System.out.println(prefix+c.getQName().getLocalName() + " : " + ((LeafListSchemaNode) c).getType().getQName());
		} else if (c instanceof ContainerSchemaNode) {
			System.out.println("!c"+prefix+c.getQName().getLocalName() + " : " );
			//create structured data type
			expand(prefix+"  ", (ContainerSchemaNode) c);
		} else if (c instanceof ListSchemaNode) {
			System.out.println("!l"+prefix+c.getQName().getLocalName() + " : " );
			//create collection of structured data type
			expand(prefix+"  ", (ListSchemaNode) c);
		} else if (c instanceof RegularLeafEffectiveStatement) {
			System.out.println("!r"+prefix+c.getQName().getLocalName() + " : " + ((RegularLeafEffectiveStatement) c).getDeclared().getType().argument());
			//create member of reference data type?
		} else if (c instanceof EmptyLeafEffectiveStatement) {
			System.out.println("!e"+prefix+c.getQName().getLocalName() + " : " + ((EmptyLeafEffectiveStatement) c).getType().getQName());
			//create member of data type?
		} else if (c instanceof LeafrefTypeDefinition) {
			//not working?
			System.out.println(((LeafrefTypeDefinition) c).getPathStatement().getOriginalString());
		} else if (c instanceof LeafSchemaNode) {
			TypeDefinition<? extends TypeDefinition<?>> type = ((LeafSchemaNode)c).getType();
			String t = type.getQName().getLocalName();
			if (!type.getQName().getLocalName().equals("string")) {
				t = type.getQName().toString();
				System.out.println("!x"+((Default<QName,LeafStatement>) c).getDeclared().getType());
			} 
			System.out.println("!"+prefix+c.getQName().getLocalName() + " : " + t);
		} else {
			System.out.println(prefix+c.getQName().getLocalName() + " : " + c.getClass().getSimpleName());
		}
	}

}
