# Copyright 2022-2025 ETSI SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)

# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at

#     http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# This file is an original contribution from Telefonica Innovación Digital S.L.

from flask import request
from flask_restx import Namespace, Resource, fields, reqparse
from src.main import NSController
from src.api.main import Api
import json
from swagger.models.create_models import create_gpp_nrm_28541_model, create_ietf_network_slice_nbi_yang_model

tfs_ns = Namespace(
    "tfs",
    description="Operations related to transport network slices with TeraflowSDN (TFS) controller"
)

# 3GPP NRM TS28.541 Data models
gpp_network_slice_request_model = create_gpp_nrm_28541_model(tfs_ns)

# IETF draft-ietf-teas-ietf-network-slice-nbi-yang Data models

slice_ddbb_model, slice_response_model = create_ietf_network_slice_nbi_yang_model(tfs_ns)

upload_parser = reqparse.RequestParser()
upload_parser.add_argument('file', location='files', type='FileStorage', help="File to upload")
upload_parser.add_argument('json_data', location='form', help="JSON Data in string format")

# Namespace Controllers
@tfs_ns.route("/slice")
class TfsSliceList(Resource):
    @tfs_ns.doc(summary="Return all transport network slices", description="Returns all transport network slices from the slice controller.")
    @tfs_ns.response(200, "Slices returned", slice_ddbb_model)
    @tfs_ns.response(404, "Transport network slices not found")
    @tfs_ns.response(500, "Internal server error")
    def get(self):
        """Retrieve all slices"""
        controller = NSController(controller_type="TFS")
        data, code = Api(controller).get_flows()
        return data, code
    
    @tfs_ns.doc(summary="Submit a transport network slice request", description="This endpoint allows clients to submit transport network slice requests using a JSON payload.")
    @tfs_ns.response(201,"Slice created successfully", slice_response_model)
    @tfs_ns.response(200, "No service to process.")
    @tfs_ns.response(400, "Invalid request format")
    @tfs_ns.response(500, "Internal server error")
    @tfs_ns.expect(upload_parser)
    def post(self):
        """Submit a new slice request with a file"""

        json_data = None

        # Try to get the JSON data from the uploaded file
        uploaded_file = request.files.get('file')
        if uploaded_file:
            if not uploaded_file.filename.endswith('.json'):
                return {
                    "success": False,
                    "data": None,
                    "error": "Only JSON files allowed"
                }, 400
            
            try:
                json_data = json.load(uploaded_file)  # Convert file to JSON
            except json.JSONDecodeError:
                return {
                    "success": False,
                    "data": None,
                    "error": "JSON file not valid"
                }, 400

        # If no file was uploaded, try to get the JSON data from the form
        if json_data is None:
            raw_json = request.form.get('json_data')
            if raw_json:
                try:
                    json_data = json.loads(raw_json)  # Convert string to JSON
                except json.JSONDecodeError:
                    return {
                        "success": False,
                        "data": None,
                        "error": "JSON file not valid"
                    }, 400
        
        # If no JSON data was found, return an error
        if json_data is None:
            return {
                    "success": False,
                    "data": None,
                    "error": "No data sent"
                }, 400

        # Process the JSON data with the NSController
        controller = NSController(controller_type="TFS")
        data, code = Api(controller).add_flow(json_data)
        return data, code
    
    @tfs_ns.doc(summary="Delete all transport network slices", description="Deletes all transport network slices from the slice controller.")
    @tfs_ns.response(204, "All transport network slices deleted successfully.")
    @tfs_ns.response(500, "Internal server error")
    def delete(self):
        """Delete all slices"""
        controller = NSController(controller_type="TFS")
        data, code = Api(controller).delete_flows()
        return data, code


@tfs_ns.route("/slice/<string:slice_id>")
@tfs_ns.doc(params={"slice_id": "The ID of the slice to retrieve or modify"})
class TfsSlice(Resource):
    @tfs_ns.doc(summary="Return a specific transport network slice", description="Returns specific information related to a slice by providing its id")
    @tfs_ns.response(200, "Slice returned", slice_ddbb_model)
    @tfs_ns.response(404, "Transport network slice not found.")
    @tfs_ns.response(500, "Internal server error")
    def get(self, slice_id):
        """Retrieve a specific slice"""
        controller = NSController(controller_type="TFS")
        data, code = Api(controller).get_flows(slice_id)
        return data, code

    @tfs_ns.doc(summary="Delete a specific transport network slice", description="Deletes a specific transport network slice from the slice controller based on the provided `slice_id`.")
    @tfs_ns.response(204, "Transport network slice deleted successfully.")
    @tfs_ns.response(404, "Transport network slice not found.")
    @tfs_ns.response(500, "Internal server error")
    def delete(self, slice_id):
        """Delete a slice"""
        controller = NSController(controller_type="TFS")
        data, code = Api(controller).delete_flows(slice_id)
        return data, code

    @tfs_ns.expect(slice_ddbb_model, validate=True)
    @tfs_ns.doc(summary="Modify a specific transport network slice", description="Returns a specific slice that has been modified")
    @tfs_ns.response(200, "Slice modified", slice_response_model)
    @tfs_ns.response(404, "Transport network slice not found.")
    @tfs_ns.response(500, "Internal server error")
    def put(self, slice_id):
        """Modify a slice"""
        json_data = request.get_json()
        controller = NSController(controller_type="TFS")
        data, code = Api(controller).modify_flow(slice_id, json_data)
        return data, code


