import pytest
import json
from pathlib import Path
from itertools import product
from src.api.main import Api
from src.main import NSController
from app import create_app 

# Carpeta donde están los JSON de requests
REQUESTS_DIR = Path(__file__).parent / "requests"

# Lista de todos los flags booleanos que quieres probar
FLAGS_TO_TEST = ["WEBUI_DEPLOY", "DUMP_TEMPLATES", "PLANNER_ENABLED", "PCE_EXTERNAL", "NRP_ENABLED"]

# Valores posibles para PLANNER_TYPE
PLANNER_TYPE_VALUES = ["ENERGY", "HRAT", "TFS_OPTICAL"]


@pytest.fixture
def app(temp_sqlite_db):
    """Crea la app Flask con configuración por defecto."""
    app = create_app()
    return app

@pytest.fixture
def client(app):
    """Cliente de test de Flask para hacer requests."""
    return app.test_client()

@pytest.fixture
def set_flags(app):
    """Cambia directamente los flags en app.config"""
    def _set(flags: dict):
        for k, v in flags.items():
            app.config[k] = v
    return _set

@pytest.fixture
def temp_sqlite_db(monkeypatch, tmp_path):
    """Usa una base de datos SQLite temporal durante los tests."""
    temp_db_path = tmp_path / "test_slice.db"
    monkeypatch.setattr("src.database.db.DB_NAME", str(temp_db_path))

    # Inicializa la base de datos temporal
    from src.database.db import init_db
    init_db()

    yield temp_db_path

    # Limpieza al finalizar
    if temp_db_path.exists():
        temp_db_path.unlink()

# Función para cargar todos los JSONs
def load_request_files():
    test_cases = []
    for f in REQUESTS_DIR.glob("*.json"):
        with open(f, "r") as file:
            json_data = json.load(file)
        test_cases.append(json_data)
    return test_cases

# Generador de todas las combinaciones de flags
def generate_flag_combinations():
    bool_values = [True, False]
    for combo in product(bool_values, repeat=len(FLAGS_TO_TEST)):
        bool_flags = dict(zip(FLAGS_TO_TEST, combo)) 
        for planner_type in PLANNER_TYPE_VALUES:
            yield {**bool_flags, "PLANNER_TYPE": planner_type}


# Fixture que combina cada request con cada combinación de flags
def generate_test_cases():
    requests = load_request_files()
    for json_data in requests:
        for flags in generate_flag_combinations():
            expected_codes = [200,201]
            yield (json_data, flags, expected_codes)

@pytest.mark.parametrize(
    "json_data, flags, expected_codes",
    list(generate_test_cases())
)
def test_add_and_delete_flow(app, json_data, flags, expected_codes, set_flags, temp_sqlite_db):
    with app.app_context():
        set_flags(flags)

        controller = NSController(controller_type="TFS")
        api = Api(controller)

        # Añadir flujo
        data, code = api.add_flow(json_data)
        assert code in expected_codes, f"Flags en fallo: {flags}"

        # Eliminar flujo si fue creado
        if code == 201 and isinstance(data, dict) and "slice_id" in data:
            slice_id = data["slice_id"]
            _, delete_code = api.delete_flows(slice_id=slice_id)
            assert delete_code == 204, f"No se pudo eliminar el slice {slice_id}"


