import pytest
import sqlite3
import json
import os
import time
from unittest.mock import patch, MagicMock
from src.database.db import (
    init_db,
    save_data,
    update_data,
    delete_data,
    get_data,
    get_all_data,
    delete_all_data,
    DB_NAME
)
from src.database.store_data import store_data


@pytest.fixture
def test_db(tmp_path):
    """Fixture to create and cleanup test database."""
    test_db_name = str(tmp_path / "test_slice.db")
    
    # Use test database
    with patch('src.database.db.DB_NAME', test_db_name):
        conn = sqlite3.connect(test_db_name)
        cursor = conn.cursor()
        cursor.execute("""
            CREATE TABLE IF NOT EXISTS slice (
                slice_id TEXT PRIMARY KEY,
                intent TEXT NOT NULL,
                controller TEXT NOT NULL
            )
        """)
        conn.commit()
        conn.close()
        
        yield test_db_name
        
        # Cleanup - Close all connections and remove file
        try:
            # Force SQLite to release locks
            sqlite3.connect(':memory:').execute('VACUUM').close()
            
            # Wait a moment for file locks to release
            import time
            time.sleep(0.1)
            
            # Remove the file if it exists
            if os.path.exists(test_db_name):
                os.remove(test_db_name)
        except Exception as e:
            # On Windows, sometimes files are locked. Try again after a delay
            import time
            time.sleep(0.5)
            try:
                if os.path.exists(test_db_name):
                    os.remove(test_db_name)
            except:
                pass  # If it still fails, let pytest's tmp_path cleanup handle it


@pytest.fixture
def sample_intent():
    """Fixture providing sample network slice intent."""
    return {
        "ietf-network-slice-service:network-slice-services": {
            "slice-service": [{
                "id": "slice-service-12345",
                "description": "Test network slice",
                "service-tags": {"tag-type": {"value": "L2VPN"}},
                "sdps": {
                    "sdp": [{
                        "node-id": "node1",
                        "sdp-ip-address": "10.0.0.1"
                    }]
                }
            }],
            "slo-sle-templates": {
                "slo-sle-template": [{
                    "id": "profile1",
                    "slo-policy": {
                        "metric-bound": [{
                            "metric-type": "one-way-bandwidth",
                            "metric-unit": "kbps",
                            "bound": 1000
                        }]
                    }
                }]
            }
        }
    }


@pytest.fixture
def simple_intent():
    """Fixture providing simple intent for basic testing."""
    return {
        "bandwidth": "1Gbps",
        "latency": "10ms",
        "provider": "opensec"
    }


class TestInitDb:
    """Tests for database initialization."""
    
    def test_init_db_creates_table(self, tmp_path):
        """Test that init_db creates the slice table."""
        test_db = str(tmp_path / "test.db")
        
        with patch('src.database.db.DB_NAME', test_db):
            init_db()
            
            conn = sqlite3.connect(test_db)
            cursor = conn.cursor()
            cursor.execute("SELECT name FROM sqlite_master WHERE type='table' AND name='slice'")
            result = cursor.fetchone()
            conn.close()
            time.sleep(0.05)  # Brief pause for file lock release
            
            assert result is not None
            assert result[0] == 'slice'
    
    def test_init_db_creates_correct_columns(self, tmp_path):
        """Test that init_db creates table with correct columns."""
        test_db = str(tmp_path / "test.db")
        
        with patch('src.database.db.DB_NAME', test_db):
            init_db()
            
            conn = sqlite3.connect(test_db)
            cursor = conn.cursor()
            cursor.execute("PRAGMA table_info(slice)")
            columns = cursor.fetchall()
            conn.close()
            time.sleep(0.05)
            
            column_names = [col[1] for col in columns]
            assert "slice_id" in column_names
            assert "intent" in column_names
            assert "controller" in column_names
    
    def test_init_db_idempotent(self, tmp_path):
        """Test that init_db can be called multiple times without error."""
        test_db = str(tmp_path / "test.db")
        
        with patch('src.database.db.DB_NAME', test_db):
            init_db()
            init_db()  # Should not raise error
            
            conn = sqlite3.connect(test_db)
            cursor = conn.cursor()
            cursor.execute("SELECT name FROM sqlite_master WHERE type='table' AND name='slice'")
            result = cursor.fetchone()
            conn.close()
            time.sleep(0.05)
            
            assert result is not None


class TestSaveData:
    """Tests for save_data function."""
    
    def test_save_data_success(self, test_db, simple_intent):
        """Test successful data saving."""
        with patch('src.database.db.DB_NAME', test_db):
            save_data("slice-001", simple_intent, "TFS")
            
            conn = sqlite3.connect(test_db)
            cursor = conn.cursor()
            cursor.execute("SELECT * FROM slice WHERE slice_id = ?", ("slice-001",))
            result = cursor.fetchone()
            conn.close()
            
            assert result is not None
            assert result[0] == "slice-001"
            assert result[2] == "TFS"
            assert json.loads(result[1]) == simple_intent
    
    def test_save_data_with_complex_intent(self, test_db, sample_intent):
        """Test saving complex nested intent structure."""
        with patch('src.database.db.DB_NAME', test_db):
            slice_id = sample_intent["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["id"]
            save_data(slice_id, sample_intent, "IXIA")
            
            conn = sqlite3.connect(test_db)
            cursor = conn.cursor()
            cursor.execute("SELECT intent FROM slice WHERE slice_id = ?", (slice_id,))
            result = cursor.fetchone()
            conn.close()
            
            retrieved_intent = json.loads(result[0])
            assert retrieved_intent == sample_intent
    
    def test_save_data_duplicate_slice_id_raises_error(self, test_db, simple_intent):
        """Test that saving duplicate slice_id raises ValueError."""
        with patch('src.database.db.DB_NAME', test_db):
            save_data("slice-001", simple_intent, "TFS")
            
            with pytest.raises(ValueError, match="already exists"):
                save_data("slice-001", simple_intent, "TFS")
    
    def test_save_data_multiple_slices(self, test_db, simple_intent):
        """Test saving multiple different slices."""
        with patch('src.database.db.DB_NAME', test_db):
            save_data("slice-001", simple_intent, "TFS")
            save_data("slice-002", simple_intent, "IXIA")
            
            conn = sqlite3.connect(test_db)
            cursor = conn.cursor()
            cursor.execute("SELECT COUNT(*) FROM slice")
            count = cursor.fetchone()[0]
            conn.close()
            
            assert count == 2
    
    def test_save_data_with_different_controllers(self, test_db, simple_intent):
        """Test saving data with different controller types."""
        with patch('src.database.db.DB_NAME', test_db):
            save_data("slice-tfs", simple_intent, "TFS")
            save_data("slice-ixia", simple_intent, "IXIA")
            
            conn = sqlite3.connect(test_db)
            cursor = conn.cursor()
            cursor.execute("SELECT controller FROM slice WHERE slice_id = ?", ("slice-tfs",))
            tfs_result = cursor.fetchone()
            cursor.execute("SELECT controller FROM slice WHERE slice_id = ?", ("slice-ixia",))
            ixia_result = cursor.fetchone()
            conn.close()
            
            assert tfs_result[0] == "TFS"
            assert ixia_result[0] == "IXIA"


class TestUpdateData:
    """Tests for update_data function."""
    
    def test_update_data_success(self, test_db, simple_intent):
        """Test successful data update."""
        with patch('src.database.db.DB_NAME', test_db):
            save_data("slice-001", simple_intent, "TFS")
            
            updated_intent = {"bandwidth": "2Gbps", "latency": "5ms", "provider": "opensec"}
            update_data("slice-001", updated_intent, "TFS")
            
            conn = sqlite3.connect(test_db)
            cursor = conn.cursor()
            cursor.execute("SELECT intent FROM slice WHERE slice_id = ?", ("slice-001",))
            result = cursor.fetchone()
            conn.close()
            
            retrieved_intent = json.loads(result[0])
            assert retrieved_intent == updated_intent
    
    def test_update_data_nonexistent_slice_raises_error(self, test_db, simple_intent):
        """Test that updating nonexistent slice raises ValueError."""
        with patch('src.database.db.DB_NAME', test_db):
            with pytest.raises(ValueError, match="No slice found"):
                update_data("nonexistent-slice", simple_intent, "TFS")
    
    def test_update_data_controller_type(self, test_db, simple_intent):
        """Test updating controller type."""
        with patch('src.database.db.DB_NAME', test_db):
            save_data("slice-001", simple_intent, "TFS")
            update_data("slice-001", simple_intent, "IXIA")
            
            conn = sqlite3.connect(test_db)
            cursor = conn.cursor()
            cursor.execute("SELECT controller FROM slice WHERE slice_id = ?", ("slice-001",))
            result = cursor.fetchone()
            conn.close()
            
            assert result[0] == "IXIA"
    
    def test_update_data_complex_intent(self, test_db, sample_intent):
        """Test updating with complex nested structure."""
        with patch('src.database.db.DB_NAME', test_db):
            slice_id = sample_intent["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["id"]
            save_data(slice_id, sample_intent, "TFS")
            
            updated_sample = sample_intent.copy()
            updated_sample["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["description"] = "Updated description"
            
            update_data(slice_id, updated_sample, "IXIA")
            
            retrieved = get_data(slice_id)
            assert retrieved["intent"]["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["description"] == "Updated description"
            assert retrieved["controller"] == "IXIA"


class TestDeleteData:
    """Tests for delete_data function."""
    
    def test_delete_data_success(self, test_db, simple_intent):
        """Test successful data deletion."""
        with patch('src.database.db.DB_NAME', test_db):
            save_data("slice-001", simple_intent, "TFS")
            delete_data("slice-001")
            
            conn = sqlite3.connect(test_db)
            cursor = conn.cursor()
            cursor.execute("SELECT * FROM slice WHERE slice_id = ?", ("slice-001",))
            result = cursor.fetchone()
            conn.close()
            
            assert result is None
    
    def test_delete_data_nonexistent_slice_raises_error(self, test_db):
        """Test that deleting nonexistent slice raises ValueError."""
        with patch('src.database.db.DB_NAME', test_db):
            with pytest.raises(ValueError, match="No slice found"):
                delete_data("nonexistent-slice")
    
    def test_delete_data_multiple_slices(self, test_db, simple_intent):
        """Test deleting one slice doesn't affect others."""
        with patch('src.database.db.DB_NAME', test_db):
            save_data("slice-001", simple_intent, "TFS")
            save_data("slice-002", simple_intent, "IXIA")
            
            delete_data("slice-001")
            
            conn = sqlite3.connect(test_db)
            cursor = conn.cursor()
            cursor.execute("SELECT COUNT(*) FROM slice")
            count = cursor.fetchone()[0]
            cursor.execute("SELECT * FROM slice WHERE slice_id = ?", ("slice-002",))
            remaining = cursor.fetchone()
            conn.close()
            
            assert count == 1
            assert remaining[0] == "slice-002"


class TestGetData:
    """Tests for get_data function."""
    
    def test_get_data_success(self, test_db, simple_intent):
        """Test retrieving existing data."""
        with patch('src.database.db.DB_NAME', test_db):
            save_data("slice-001", simple_intent, "TFS")
            result = get_data("slice-001")
            
            assert result["slice_id"] == "slice-001"
            assert result["intent"] == simple_intent
            assert result["controller"] == "TFS"
    
    def test_get_data_nonexistent_raises_error(self, test_db):
        """Test that getting nonexistent slice raises ValueError."""
        with patch('src.database.db.DB_NAME', test_db):
            with pytest.raises(ValueError, match="No slice found"):
                get_data("nonexistent-slice")
    
    def test_get_data_json_parsing(self, test_db, sample_intent):
        """Test that returned intent is parsed JSON."""
        with patch('src.database.db.DB_NAME', test_db):
            slice_id = sample_intent["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["id"]
            save_data(slice_id, sample_intent, "TFS")
            result = get_data(slice_id)
            
            assert isinstance(result["intent"], dict)
            assert result["intent"] == sample_intent
    
    def test_get_data_returns_all_fields(self, test_db, simple_intent):
        """Test that get_data returns all fields."""
        with patch('src.database.db.DB_NAME', test_db):
            save_data("slice-001", simple_intent, "TFS")
            result = get_data("slice-001")
            
            assert "slice_id" in result
            assert "intent" in result
            assert "controller" in result
            assert len(result) == 3


class TestGetAllData:
    """Tests for get_all_data function."""
    
    def test_get_all_data_empty_database(self, test_db):
        """Test retrieving all data from empty database."""
        with patch('src.database.db.DB_NAME', test_db):
            result = get_all_data()
            assert result == []
    
    def test_get_all_data_single_slice(self, test_db, simple_intent):
        """Test retrieving all data with single slice."""
        with patch('src.database.db.DB_NAME', test_db):
            save_data("slice-001", simple_intent, "TFS")
            result = get_all_data()
            
            assert len(result) == 1
            assert result[0]["slice_id"] == "slice-001"
            assert result[0]["intent"] == simple_intent
    
    def test_get_all_data_multiple_slices(self, test_db, simple_intent):
        """Test retrieving all data with multiple slices."""
        with patch('src.database.db.DB_NAME', test_db):
            save_data("slice-001", simple_intent, "TFS")
            save_data("slice-002", simple_intent, "IXIA")
            save_data("slice-003", simple_intent, "TFS")
            
            result = get_all_data()
            
            assert len(result) == 3
            slice_ids = [slice_data["slice_id"] for slice_data in result]
            assert "slice-001" in slice_ids
            assert "slice-002" in slice_ids
            assert "slice-003" in slice_ids
    
    def test_get_all_data_json_parsing(self, test_db, sample_intent):
        """Test that all returned intents are parsed JSON."""
        with patch('src.database.db.DB_NAME', test_db):
            slice_id = sample_intent["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["id"]
            save_data(slice_id, sample_intent, "TFS")
            save_data("slice-002", sample_intent, "IXIA")
            
            result = get_all_data()
            
            for slice_data in result:
                assert isinstance(slice_data["intent"], dict)
    
    def test_get_all_data_includes_all_controllers(self, test_db, simple_intent):
        """Test that get_all_data includes slices from different controllers."""
        with patch('src.database.db.DB_NAME', test_db):
            save_data("slice-tfs", simple_intent, "TFS")
            save_data("slice-ixia", simple_intent, "IXIA")
            
            result = get_all_data()
            
            controllers = [slice_data["controller"] for slice_data in result]
            assert "TFS" in controllers
            assert "IXIA" in controllers


class TestDeleteAllData:
    """Tests for delete_all_data function."""
    
    def test_delete_all_data_removes_all_slices(self, test_db, simple_intent):
        """Test that delete_all_data removes all slices."""
        with patch('src.database.db.DB_NAME', test_db):
            save_data("slice-001", simple_intent, "TFS")
            save_data("slice-002", simple_intent, "IXIA")
            
            delete_all_data()
            
            result = get_all_data()
            assert result == []
    
    def test_delete_all_data_empty_database(self, test_db):
        """Test delete_all_data on empty database doesn't raise error."""
        with patch('src.database.db.DB_NAME', test_db):
            delete_all_data()  # Should not raise error
            result = get_all_data()
            assert result == []


class TestStoreData:
    """Tests for store_data wrapper function."""
    
    def test_store_data_save_new_slice(self, test_db, sample_intent):
        """Test store_data saves new slice without slice_id."""
        with patch('src.database.db.DB_NAME', test_db):
            store_data(sample_intent, None, "TFS")
            
            slice_id = sample_intent["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["id"]
            result = get_data(slice_id)
            
            assert result["slice_id"] == slice_id
            assert result["intent"] == sample_intent
            assert result["controller"] == "TFS"
    
    def test_store_data_update_existing_slice(self, test_db, sample_intent):
        """Test store_data updates existing slice when slice_id provided."""
        with patch('src.database.db.DB_NAME', test_db):
            slice_id = sample_intent["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["id"]
            
            # Save initial data
            save_data(slice_id, sample_intent, "TFS")
            
            # Update with store_data
            updated_intent = sample_intent.copy()
            updated_intent["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["description"] = "Updated"
            store_data(updated_intent, slice_id, "IXIA")
            
            result = get_data(slice_id)
            assert result["controller"] == "IXIA"
            assert result["intent"]["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["description"] == "Updated"
    
    def test_store_data_extracts_slice_id_from_intent(self, test_db, sample_intent):
        """Test store_data correctly extracts slice_id from intent structure."""
        with patch('src.database.db.DB_NAME', test_db):
            store_data(sample_intent, None, "TFS")
            
            all_data = get_all_data()
            assert len(all_data) == 1
            assert all_data[0]["slice_id"] == "slice-service-12345"
    
    def test_store_data_with_different_controllers(self, test_db, sample_intent):
        """Test store_data works with different controller types."""
        with patch('src.database.db.DB_NAME', test_db):
            store_data(sample_intent, None, "TFS")
            
            slice_id = sample_intent["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["id"]
            result = get_data(slice_id)
            
            assert result["controller"] == "TFS"


class TestDatabaseIntegration:
    """Integration tests for database operations."""
    
    def test_full_lifecycle_create_read_update_delete(self, test_db, simple_intent):
        """Test complete slice lifecycle."""
        with patch('src.database.db.DB_NAME', test_db):
            # Create
            save_data("slice-lifecycle", simple_intent, "TFS")
            
            # Read
            result = get_data("slice-lifecycle")
            assert result["slice_id"] == "slice-lifecycle"
            
            # Update
            updated_intent = {"bandwidth": "5Gbps", "latency": "2ms", "provider": "opensec"}
            update_data("slice-lifecycle", updated_intent, "IXIA")
            
            result = get_data("slice-lifecycle")
            assert result["intent"] == updated_intent
            assert result["controller"] == "IXIA"
            
            # Delete
            delete_data("slice-lifecycle")
            
            with pytest.raises(ValueError):
                get_data("slice-lifecycle")
    
    def test_concurrent_operations(self, test_db, simple_intent):
        """Test multiple concurrent database operations."""
        with patch('src.database.db.DB_NAME', test_db):
            # Create multiple slices
            for i in range(5):
                save_data(f"slice-{i}", simple_intent, "TFS" if i % 2 == 0 else "IXIA")
            
            # Verify all created
            all_data = get_all_data()
            assert len(all_data) == 5
            
            # Update some
            updated_intent = {"updated": True}
            for i in range(0, 3):
                update_data(f"slice-{i}", updated_intent, "TFS")
            
            # Verify updates
            for i in range(0, 3):
                result = get_data(f"slice-{i}")
                assert result["intent"]["updated"] is True
            
            # Delete some
            delete_data("slice-0")
            delete_data("slice-2")
            
            all_data = get_all_data()
            assert len(all_data) == 3
    
    def test_data_persistence_across_operations(self, test_db, sample_intent):
        """Test that data persists correctly across multiple operations."""
        with patch('src.database.db.DB_NAME', test_db):
            slice_id = sample_intent["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["id"]
            
            # Save
            save_data(slice_id, sample_intent, "TFS")
            
            # Get all and verify
            all_before = get_all_data()
            assert len(all_before) == 1
            
            # Save another
            save_data("slice-other", sample_intent, "IXIA")
            all_after = get_all_data()
            assert len(all_after) == 2
            
            # Verify first slice still intact
            first_slice = get_data(slice_id)
            assert first_slice["intent"] == sample_intent
            assert first_slice["controller"] == "TFS"