# Copyright 2022-2025 ETSI SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)

# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at

#     http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# This file includes original contributions from Telefonica Innovación Digital S.L.

import logging, os
from src.config.constants import TEMPLATES_PATH, NBI_L3_PATH
from src.utils.load_template import load_template
from src.utils.safe_get import safe_get
from flask import current_app

def tfs_l3vpn(ietf_intent, response):
    """
    Translate L3VPN (Layer 3 Virtual Private Network) intent into a TeraFlow service request.

    Similar to __tfs_l2vpn, but configured for Layer 3 VPN:
    1. Defines endpoint routers
    2. Loads service template
    3. Generates unique service UUID
    4. Configures service endpoints
    5. Adds QoS constraints
    6. Prepares configuration rules for network interfaces

    Args:
        ietf_intent (dict): IETF-formatted network slice intent.
        response (dict): Response data containing slice information.

    Returns:
        dict: A TeraFlow service request for L3VPN configuration.
    """
    # Hardcoded router endpoints
    # TODO (should be dynamically determined)
    origin_router_id = safe_get(ietf_intent, ["ietf-network-slice-service:network-slice-services", "slice-service", 0, "sdps", "sdp", 0, "attachment-circuits", "attachment-circuit", 0, "sdp-peering", "peer-sap-id"])
    if not origin_router_id:
        logging.warning("Origin router ID not found in the intent. Skipping L3VPN realization.")
        return None
    origin_router_if = '0/0/0-GigabitEthernet0/0/0/0'
    destination_router_id = safe_get(ietf_intent, ["ietf-network-slice-service:network-slice-services", "slice-service", 0, "sdps", "sdp", 1, "attachment-circuits", "attachment-circuit", 0, "sdp-peering", "peer-sap-id"])
    if not destination_router_id:
        logging.warning("Destination router ID not found in the intent. Skipping L3VPN realization.")
        return None
    destination_router_if = '0/0/0-GigabitEthernet0/0/0/0'
    id = ietf_intent["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["id"]
    slice = next((d for d in response if d.get("id") == id), None)

    if current_app.config["UPLOAD_TYPE"] == "WEBUI":
        # Load L3VPN service template
        tfs_request = load_template(os.path.join(TEMPLATES_PATH, "L3-VPN_template_empty.json"))["services"][0]
        # Configure service UUID
        tfs_request["service_id"]["service_uuid"]["uuid"] = ietf_intent['ietf-network-slice-service:network-slice-services']['slice-service'][0]["id"]

        # Configure service endpoints
        for endpoint in tfs_request["service_endpoint_ids"]:
            endpoint["device_id"]["device_uuid"]["uuid"] = origin_router_id if endpoint is tfs_request["service_endpoint_ids"][0] else destination_router_id
            endpoint["endpoint_uuid"]["uuid"] = origin_router_if if endpoint is tfs_request["service_endpoint_ids"][0] else destination_router_if

        # Add service constraints
        for constraint in slice.get("requirements", []):
            tfs_request["service_constraints"].append({"custom": constraint})

        # Add configuration rules
        for i, config_rule in enumerate(tfs_request["service_config"]["config_rules"][1:], start=1):
            router_id = origin_router_id if i == 1 else destination_router_id
            router_if = origin_router_if if i == 1 else destination_router_if
            resource_value = config_rule["custom"]["resource_value"]

            sdp_index = i - 1
            vlan_value = ietf_intent["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["sdps"]["sdp"][sdp_index]["service-match-criteria"]["match-criterion"][0]["value"]
            resource_value["router_id"] = destination_router_id if i == 1 else origin_router_id
            resource_value["vlan_id"] = int(vlan_value)
            resource_value["address_ip"] = destination_router_id if i == 1 else origin_router_id
            resource_value["policy_AZ"] = "policyA"
            resource_value["policy_ZA"] = "policyB"
            resource_value["ni_name"] = 'ELAN{:s}'.format(str(vlan_value))
            config_rule["custom"]["resource_key"] = f"/device[{router_id}]/endpoint[{router_if}]/settings"
    
    elif current_app.config["UPLOAD_TYPE"] == "NBI":
        #self.path = NBI_L3_PATH

        # Load IETF L3VPN service template
        tfs_request =  load_template(os.path.join(TEMPLATES_PATH, "ietfL3VPN_template_empty.json"))

        # Add path to the request
        tfs_request["path"] = NBI_L3_PATH

        # Generate service UUID
        full_id = ietf_intent["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["id"]
        tfs_request["ietf-l3vpn-svc:l3vpn-svc"]["vpn-services"]["vpn-service"][0]["vpn-id"] = full_id
        # Configure service endpoints
        for i, site in enumerate(tfs_request["ietf-l3vpn-svc:l3vpn-svc"]["sites"]["site"]):

            # Determine if origin or destination
            is_origin = (i == 0)
            sdp_index = 0 if is_origin else 1
            location = ietf_intent["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["sdps"]["sdp"][sdp_index]["node-id"]
            router_id = origin_router_id if is_origin else destination_router_id
            router_if = origin_router_if if is_origin else destination_router_if

            # Assign common values
            site["site-id"] = f"site_{location}"
            site["locations"]["location"][0]["location-id"] = location
            site["devices"]["device"][0]["device-id"] = router_id
            site["devices"]["device"][0]["location"] = location

            access = site["site-network-accesses"]["site-network-access"][0]
            access["site-network-access-id"] = router_if
            access["device-reference"] = router_id
            access["vpn-attachment"]["vpn-id"] = full_id

            # Aplicar restricciones QoS
            for constraint in slice.get("requirements", []):
                ctype = constraint["constraint_type"]
                cvalue = float(constraint["constraint_value"])
                if constraint["constraint_type"].startswith("one-way-bandwidth"):
                        unit = constraint["constraint_type"].split("[")[-1].rstrip("]")
                        multiplier = {"bps": 1, "kbps": 1_000, "Mbps": 1_000_000, "Gbps": 1_000_000_000}.get(unit, 1)
                        value = int(cvalue * multiplier)
                        access["service"]["svc-input-bandwidth"] = value
                        access["service"]["svc-output-bandwidth"] = value
                elif ctype == "one-way-delay-maximum[milliseconds]":
                    access["service"]["qos"]["qos-profile"]["classes"]["class"][0]["latency"]["latency-boundary"] = int(cvalue)
                elif ctype == "availability[%]":
                    access["service"]["qos"]["qos-profile"]["classes"]["class"][0]["bandwidth"]["guaranteed-bw-percent"] = int(cvalue)
                elif ctype == "mtu[bytes]":
                    access["service"]["svc-mtu"] = int(cvalue)

    
    logging.info(f"L3VPN Intent realized")
    #self.answer[self.subnet]["VLAN"] = vlan_value
    return tfs_request