# Copyright 2022-2025 ETSI SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)

# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at

#     http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# This file is an original contribution from Telefonica Innovación Digital S.L.

import logging, requests

def hrat_planner(data: str, ip: str, action: str = "create") -> dict:
    """
    Interface with the HRAT (Hierarchical Resource Allocation Tool) for transport network slice management.
    
    This function communicates with an external HRAT service to create or delete
    transport network slices, handling optical layer provisioning and IP layer
    configuration.
    
    Args:
        data (str or dict): Network slice UUID for deletion, or full intent data for creation
        ip (str): IP address of the HRAT service
        action (str, optional): Operation to perform - "create" or "delete". Defaults to "create"
    
    Returns:
        dict: Response from HRAT service containing:
            - network-slice-uuid: Unique identifier for the slice
            - viability: Boolean indicating if slice is viable
            - actions: List of configuration actions including:
                * CREATE_OPTICAL_SLICE
                * PROVISION_MEDIA_CHANNEL_OLS_PATH
                * ACTIVATE_TRANSCEIVER
                * CONFIG_VPNL3
              
    Notes:
        - On timeout or connection errors, returns static fallback data
        - HRAT service expected at port 9090
        - Timeout set to 15 seconds for all requests
        
    Raises:
        requests.exceptions.RequestException: On HTTP request failures (logged, not raised)
    """
    data_static = {'network-slice-uuid': 'ecoc25-short-path-a7764e55-9bdb-4e38-9386-02ff47a33225', 'viability': True, 'actions': [{'type': 'CREATE_OPTICAL_SLICE', 'layer': 'OPTICAL', 'content': {'tenant-uuid': 'ea4ade23-1444-4f93-aabc-4fcbe2ae74dd', 'service-interface-point': [{'uuid': 'e7444187-119b-5b2e-8a60-ee26b30c441a'}, {'uuid': 'b32b1623-1f64-59d2-8148-b035a8f77625'}], 'node': [{'uuid': '68eb48ac-b686-5653-bdaf-7ccaeecd0709', 'owned-node-edge-point': [{'uuid': '7fd74b80-2b5a-55e2-8ef7-82bf589c9591', 'media-channel-node-edge-point-spec': {'mc-pool': {'supportable-spectrum': [{'lower-frequency': '191325000', 'upper-frequency': '192225000'}, {'lower-frequency': '194325000', 'upper-frequency': '195225000'}]}}}, {'uuid': '7b9f0b65-2387-5352-bc36-7173639463f0', 'media-channel-node-edge-point-spec': {'mc-pool': {'supportable-spectrum': [{'lower-frequency': '191325000', 'upper-frequency': '192225000'}, {'lower-frequency': '194325000', 'upper-frequency': '195225000'}]}}}]}, {'uuid': 'f55351ce-a5c8-50a7-b506-95b40e08bce4', 'owned-node-edge-point': [{'uuid': 'da6d924d-9cb4-5add-817d-f83e910beb2e', 'media-channel-node-edge-point-spec': {'mc-pool': {'supportable-spectrum': [{'lower-frequency': '191325000', 'upper-frequency': '192225000'}, {'lower-frequency': '194325000', 'upper-frequency': '195225000'}]}}}, {'uuid': '577ec899-ad92-5a19-a140-405a3cdbaa17', 'media-channel-node-edge-point-spec': {'mc-pool': {'supportable-spectrum': [{'lower-frequency': '191325000', 'upper-frequency': '192225000'}, {'lower-frequency': '194325000', 'upper-frequency': '195225000'}]}}}]}], 'link': [{'uuid': '3beef785-bb26-5741-af10-c5e1838c1701'}, {'uuid': '6144c664-246a-58ed-bf0a-7ec4286625da'}]}, 'controller-uuid': 'TAPI Optical Controller'}, {'type': 'PROVISION_MEDIA_CHANNEL_OLS_PATH', 'layer': 'OPTICAL', 'content': {'ols-path-uuid': 'cfeae4cb-c305-4884-9945-8b0c0f040c98', 'src-sip-uuid': 'e7444187-119b-5b2e-8a60-ee26b30c441a', 'dest-sip-uuid': 'b32b1623-1f64-59d2-8148-b035a8f77625', 'direction': 'BIDIRECTIONAL', 'layer-protocol-name': 'PHOTONIC_MEDIA', 'layer-protocol-qualifier': 'tapi-photonic-media:PHOTONIC_LAYER_QUALIFIER_MC', 'bandwidth-ghz': 100, 'link-uuid-path': ['3beef785-bb26-5741-af10-c5e1838c1701'], 'lower-frequency-mhz': '194700000', 'upper-frequency-mhz': '194800000', 'adjustment-granularity': 'G_6_25GHZ', 'grid-type': 'FLEX'}, 'controller-uuid': 'TAPI Optical Controller', 'tenant-uuid': 'ea4ade23-1444-4f93-aabc-4fcbe2ae74dd'}, {'type': 'ACTIVATE_TRANSCEIVER', 'layer': 'OPTICAL', 'content': {'node-uuid': 'Phoenix-1', 'termination-point-uuid': 'Ethernet110', 'transceiver-type': 'CFP2', 'frequency-ghz': 194700.0, 'spectrum-width-ghz': 100.0, 'tx-power-dbm': 0.0}, 'controller-uuid': 'IP Controller'}, {'type': 'ACTIVATE_TRANSCEIVER', 'layer': 'OPTICAL', 'content': {'node-uuid': 'Phoenix-2', 'termination-point-uuid': 'Ethernet220', 'transceiver-type': 'CFP2', 'frequency-ghz': 194700.0, 'spectrum-width-ghz': 100.0, 'tx-power-dbm': 0.0}, 'controller-uuid': 'IP Controller'}, {'type': 'CONFIG_VPNL3', 'layer': 'IP', 'content': {'tunnel-uuid': '9aae851a-eea9-4a28-969f-0e2c2196e936', 'src-node-uuid': 'Phoenix-1', 'src-ip-address': '10.10.1.1', 'src-ip-mask': '/24', 'src-vlan-id': 100, 'dest-node-uuid': 'Phoenix-2', 'dest-ip-address': '10.10.2.1', 'dest-ip-mask': '/24', 'dest-vlan-id': 100}, 'controller-uuid': 'IP Controller'}]}
    url = f'http://{ip}:9090/api/resource-allocation/transport-network-slice-l3'
    headers = {'Content-Type': 'application/json'}

    try:
        if action == "delete":
            # Build deletion payload with slice ID
            payload = {
                "ietf-network-slice-service:network-slice-services": {
                    "slice-service": [
                        {
                            "id": data
                        }
                    ]
                }
            }
            response = requests.delete(url, headers=headers, json=payload, timeout=1)
        elif action == "create":
            response = requests.post(url, headers=headers, json=data, timeout=1)
        else:
            logging.error("Invalid action. Use 'create' or 'delete'.")
            return data_static
        
        if response.ok:
            return response.json()
        else:
            logging.error(f"Request failed with status code {response.status_code}: {response.text}")
            return data_static

    except requests.exceptions.RequestException as e:
        logging.error(f"HTTP request failed: {e}. Returning default data")
        return data_static
    except Exception as e:
        logging.error(f"Unexpected error: {e}")
        return data_static

