# Copyright 2022-2025 ETSI SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)

# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at

#     http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# This file is an original contribution from Telefonica Innovación Digital S.L.

import logging, random, os, json, heapq  
from src.config.constants import SRC_PATH
from flask import current_app
from src.utils.safe_get import safe_get


def energy_planner(intent):
    """
    Plan an optimal network path based on energy consumption metrics.

    This function calculates the most energy-efficient path between source
    and destination nodes, considering energy consumption, carbon emissions,
    energy efficiency, and renewable energy usage constraints.

    Args:
        intent (dict): Network slice intent containing service delivery points
                      and energy-related SLO constraints

    Returns:
        list or None: Ordered list of node names representing the optimal path,
                     or None if no valid path is found or topology is not recognized

    Notes:
        - Only supports topology with nodes A through G
        - Can use external PCE or internal Dijkstra-based algorithm
        - Considers DLOS (Delay and Loss Objectives) for energy metrics:
          EC (Energy Consumption), CE (Carbon Emission), 
          EE (Energy Efficiency), URE (Renewable Energy Usage)

    Raises:
        Exception: For errors in energy metrics or topology retrieval
    """    
    energy_metrics = retrieve_energy()
    topology = retrieve_topology()
    source = safe_get(intent, ["ietf-network-slice-service:network-slice-services", "slice-service", 0, "sdps", "sdp", 0, "node-id"])
    destination = safe_get(intent, ["ietf-network-slice-service:network-slice-services", "slice-service", 0, "sdps", "sdp", 1, "node-id"])
    optimal_path = []
    allowed_ids = {"A", "B", "C", "D", "E", "F", "G"}

    if source not in allowed_ids or destination not in allowed_ids:
        logging.warning(f"Topology not recognized (source: {source}, destination: {destination}). Skipping energy-based planning.")
        return None
    
    # If using an external PCE
    if current_app.config["PCE_EXTERNAL"]:
        logging.debug("Using external PCE for path planning")    
        def build_slice_input(node_source, node_destination):
            """Build input format for external PCE slice computation."""
            return {
                "clientName": "demo-client",
                "requestId": random.randint(1000, 9999),
                "sites": [node_source["nodeId"], node_destination["nodeId"]],
                "graph": {
                    "nodes": [
                        {
                            "nodeId": node_source["nodeId"],
                            "name": node_source["name"],
                            "footprint": node_source["footprint"],
                            "sticky": [node_source["nodeId"]]
                        },
                        {
                            "nodeId": node_destination["nodeId"],
                            "name": node_destination["name"],
                            "footprint": node_destination["footprint"],
                            "sticky": [node_destination["nodeId"]]
                        }
                    ],
                    "links": [
                        {
                            "fromNodeId": node_source["nodeId"],
                            "toNodeId": node_destination["nodeId"],
                            "bandwidth": 1000000000,
                            "metrics": [
                                {
                                    "metric": "DELAY",
                                    "value": 10,
                                    "bound": True,
                                    "required": True
                                }
                            ]
                        }
                    ],
                    "constraints": {
                        "maxVulnerability": 3,
                        "maxDeployedServices": 10,
                        "metricLimits": []
                    }
                }
            }
        
        source = next((node for node in topology["nodes"] if node["name"] == source), None)
        destination = next((node for node in topology["nodes"] if node["name"] == destination), None)
        slice_input = build_slice_input(source, destination)

        def simulate_slice_output(input_data):
            """
            Simulate external PCE response for slice computation.
            
            Args:
                input_data (dict): Input data for slice computation
                
            Returns:
                dict: Simulated slice output with path information
            """
            return {
                "input": input_data,
                "slice": {
                    "nodes": [
                        {"site": 1, "service": 1},
                        {"site": 2, "service": 2}
                    ],
                    "links": [
                        {
                            "fromNodeId": 1,
                            "toNodeId": 2,
                            "lspId": 500,
                            "path": {
                                "ingressNodeId": 1,
                                "egressNodeId": 2,
                                "hops": [
                                    {"nodeId": 3, "linkId": "A-C", "portId": 1},
                                    {"nodeId": 2, "linkId": "C-B", "portId": 2}
                                ]
                            }
                        }
                    ],
                    "metric": {"value": 9}
                },
                "error": None
            }
        
        slice_output = simulate_slice_output(slice_input)
        # Build optimal path from PCE response
        optimal_path.append(source["name"])
        for link in slice_output["slice"]["links"]:
            for hop in link["path"]["hops"]:
                optimal_path.append(next((node for node in topology["nodes"] if node["nodeId"] == hop['nodeId']), None)["name"])
    
    else:
        logging.debug("Using internal PCE for path planning")
        ietf_dlos = intent["ietf-network-slice-service:network-slice-services"]["slo-sle-templates"]["slo-sle-template"][0]["slo-policy"]["metric-bound"]
        logging.debug(ietf_dlos)
        
        # Extract DLOS (Delay and Loss Objectives) constraints
        dlos = {
            "EC": next((item.get("bound") for item in ietf_dlos if item.get("metric-type") == "energy_consumption"), None),
            "CE": next((item.get("bound") for item in ietf_dlos if item.get("metric-type") == "carbon_emission"), None),
            "EE": next((item.get("bound") for item in ietf_dlos if item.get("metric-type") == "energy_efficiency"), None),
            "URE": next((item.get("bound") for item in ietf_dlos if item.get("metric-type") == "renewable_energy_usage"), None)
        }
        logging.debug(f"Planning optimal path from {source} to {destination} with DLOS: {dlos}")
        optimal_path = calculate_optimal_path(topology, energy_metrics, source, destination, dlos)

    if not optimal_path:
        logging.error("No valid energy path found")
        return None

    return optimal_path


def retrieve_energy():
    """
    Retrieve energy consumption data for network nodes.
    
    Returns:
        dict: Energy metrics including power consumption, carbon emissions,
              efficiency, and renewable energy usage for each node
              
    Notes:
        TODO: Implement logic to retrieve real-time data from controller
        Currently reads from static JSON file
    """
    with open(os.path.join(SRC_PATH, "planner/energy_planner/energy_ddbb.json"), "r") as archivo:
        energy_metrics = json.load(archivo)
    return energy_metrics


def retrieve_topology():
    """
    Retrieve network topology information.
    
    Returns:
        dict: Network topology with nodes and links
        
    Notes:
        - If PCE_EXTERNAL is True, retrieves topology for external PCE format
        - Otherwise retrieves topology in internal format
        TODO: Implement logic to retrieve real-time data from controller
        Currently reads from static JSON files
    """
    if current_app.config["PCE_EXTERNAL"]:
        # TODO: Implement the logic to retrieve topology data from external PCE
        # GET /sss/v1/topology/node and /sss/v1/topology/link
        with open(os.path.join(SRC_PATH, "planner/energy_planner/ext_topo_ddbb.json"), "r") as archivo:
            topology = json.load(archivo)
    else:
        # TODO: Implement the logic to retrieve topology data from controller
        with open(os.path.join(SRC_PATH, "planner/energy_planner/topo_ddbb.json"), "r") as archivo:
            topology = json.load(archivo)
    return topology


def calculate_optimal_path(topology, energy_metrics, source, destination, dlos):
    """
    Calculate the optimal path using Dijkstra's algorithm with energy constraints.
    
    This function implements a constrained shortest path algorithm that considers
    energy consumption, carbon emissions, energy efficiency, and renewable energy
    usage as optimization criteria.
    
    Args:
        topology (dict): Network topology with nodes and links
        energy_metrics (dict): Energy consumption data for each node
        source (str): Source node identifier
        destination (str): Destination node identifier
        dlos (dict): Constraint bounds for:
                    - EC: Energy Consumption limit
                    - CE: Carbon Emission limit
                    - EE: Energy Efficiency limit
                    - URE: Minimum Renewable Energy Usage
    
    Returns:
        list: Ordered list of node names forming the optimal path,
              or empty list if no valid path exists
              
    Notes:
        - Uses modified Dijkstra's algorithm with multiple constraints
        - Paths violating any DLOS constraint are discarded
        - Node weights computed using compute_node_weight function
    """
    logging.debug("Starting optimal path calculation...")
    
    # Create a dictionary with the weights of each node
    node_data_map = {}
    for node_data in energy_metrics:
        node_id = node_data["name"]
        ec = node_data["typical-power"]
        ce = node_data["carbon-emissions"]
        ee = node_data["efficiency"]
        ure = node_data["renewable-energy-usage"]

        total_power_supply = sum(ps["typical-power"] for ps in node_data["power-supply"])
        total_power_boards = sum(b["typical-power"] for b in node_data["boards"])
        total_power_components = sum(c["typical-power"] for c in node_data["components"])
        total_power_transceivers = sum(t["typical-power"] for t in node_data["transceivers"])

        logging.debug(f"Node {node_id}: EC={ec}, CE={ce}, EE={ee}, URE={ure}")
        logging.debug(f"Node {node_id}: PS={total_power_supply}, BO={total_power_boards}, CO={total_power_components}, TR={total_power_transceivers}")

        weight = compute_node_weight(ec, ce, ee, ure,
                                            total_power_supply,
                                            total_power_boards,
                                            total_power_components,
                                            total_power_transceivers)
        logging.debug(f"Weight for node {node_id}: {weight}")
        
        node_data_map[node_id] = {
            "weight": weight,
            "ec": ec,
            "ce": ce,
            "ee": ee,
            "ure": ure
        }

    # Create a graph representation of the topology
    graph = {}
    for node in topology["ietf-network:networks"]["network"][0]["node"]:
        graph[node["node-id"]] = []
    for link in topology["ietf-network:networks"]["network"][0]["link"]:
        src = link["source"]["source-node"]
        dst = link["destination"]["dest-node"]
        graph[src].append((dst, node_data_map[dst]["weight"]))
        logging.debug(f"Added link: {src} -> {dst} with weight {node_data_map[dst]['weight']}")

    # Dijkstra's algorithm with restrictions
    # Queue: (accumulated cost, current node, path, sum_ec, sum_ce, sum_ee, min_ure)
    queue = [(0, source, [], 0, 0, 0, 1)]
    visited = set()

    logging.debug(f"Starting search from {source} to {destination} with restrictions: {dlos}")

    while queue:
        cost, node, path, sum_ec, sum_ce, sum_ee, min_ure = heapq.heappop(queue)
        logging.debug(f"Exploring node {node} with cost {cost} and path {path + [node]}")
        
        if node in visited:
            logging.debug(f"Node {node} already visited, skipped.")
            continue
        visited.add(node)
        path = path + [node]

        node_metrics = node_data_map[node]
        sum_ec += node_metrics["ec"]
        sum_ce += node_metrics["ce"]
        sum_ee += node_metrics["ee"]
        min_ure = min(min_ure, node_metrics["ure"]) if path[:-1] else node_metrics["ure"]

        logging.debug(f"Accumulated -> EC: {sum_ec}, CE: {sum_ce}, EE: {sum_ee}, URE min: {min_ure}")

        # Check constraint violations
        if dlos["EC"] is not None and sum_ec > dlos["EC"]:
            logging.debug(f"Discarded path {path} for exceeding EC ({sum_ec} > {dlos['EC']})")
            continue
        if dlos["CE"] is not None and sum_ce > dlos["CE"]:
            logging.debug(f"Discarded path {path} for exceeding CE ({sum_ce} > {dlos['CE']})")
            continue
        if dlos["EE"] is not None and sum_ee > dlos["EE"]:
            logging.debug(f"Discarded path {path} for exceeding EE ({sum_ee} > {dlos['EE']})")
            continue
        if dlos["URE"] is not None and min_ure < dlos["URE"]:
            logging.debug(f"Discarded path {path} for not reaching minimum URE ({min_ure} < {dlos['URE']})")
            continue

        if node == destination:
            logging.debug(f"Destination {destination} reached with a valid path: {path}")
            return path

        for neighbor, weight in graph.get(node, []):
            if neighbor not in visited:
                logging.debug(f"Queue -> neighbour: {neighbor}, weight: {weight}")
                heapq.heappush(queue, (
                    cost + weight,
                    neighbor,
                    path,
                    sum_ec,
                    sum_ce,
                    sum_ee,
                    min_ure
                ))
    
    logging.debug("No valid path found that meets the restrictions.")
    return []


def compute_node_weight(ec, ce, ee, ure, total_power_supply, total_power_boards, 
                       total_power_components, total_power_transceivers, 
                       alpha=1, beta=1, gamma=1, delta=1):
    """
    Calculate node weight based on energy and environmental metrics.
    
    Computes a green index that represents the environmental impact of routing
    traffic through a node, considering power consumption and carbon emissions.
    
    Args:
        ec (float): Base energy consumption of the node
        ce (float): Carbon emissions factor
        ee (float): Energy efficiency metric
        ure (float): Renewable energy usage ratio (0-1)
        total_power_supply (float): Total power from supply units
        total_power_boards (float): Total power consumed by boards
        total_power_components (float): Total power consumed by components
        total_power_transceivers (float): Total power consumed by transceivers
        alpha (float, optional): Weight for energy consumption. Defaults to 1
        beta (float, optional): Weight for carbon emissions. Defaults to 1
        gamma (float, optional): Weight for energy efficiency. Defaults to 1
        delta (float, optional): Weight for renewable energy. Defaults to 1
    
    Returns:
        float: Computed green index representing environmental impact
        
    Notes:
        Formula: green_index = (power_idle + power_traffic) * time / 1000 * (1 - ure) * ce
        - Assumes 100 units of traffic
        - Measured over 1 hour time period
    """
    traffic = 100
    # Measure one hour of traffic
    time = 1

    power_idle = ec + total_power_supply + total_power_boards + total_power_components + total_power_transceivers
    power_traffic = traffic * ee

    power_total = (power_idle + power_traffic)

    green_index = power_total * time / 1000 * (1 - ure) * ce

    return green_index