# Copyright 2022-2025 ETSI SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)

# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at

#     http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# This file is an original contribution from Telefonica Innovación Digital S.L.

import logging
from .detect_format import detect_format
from .translator import translator

def nbi_processor(intent_json):
    """
    Process and translate network slice intents from different formats (3GPP or IETF).

    This method detects the input JSON format and converts 3GPP intents to IETF format.

    Args:
        intent_json (dict): Input network slice intent in either 3GPP or IETF format.

    Returns:
        list: A list of IETF-formatted network slice intents.

    Raises:
        ValueError: If the JSON request format is not recognized.
    """
    # Detect the input JSON format (3GPP or IETF)
    format = detect_format(intent_json)
    ietf_intents = []

    # TODO Needs to be generalized to support different names of slicesubnets
    # Process different input formats
    if format == "3GPP":
        # Translate each subnet in 3GPP format to IETF format
        for subnet in intent_json["RANSliceSubnet1"]["networkSliceSubnetRef"]:
            ietf_intents.append(translator(intent_json, subnet))
        logging.info(f"3GPP requests translated to IETF template")
    elif format == "IETF":
        # If already in IETF format, add directly
        logging.info(f"IETF intent received")
        ietf_intents.append(intent_json)
    else:
        # Handle unrecognized format
        logging.error(f"JSON request format not recognized")
        raise ValueError("JSON request format not recognized")
    
    return ietf_intents or None