# Copyright 2022-2025 ETSI SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)

# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at

#     http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# This file is an original contribution from Telefonica Innovación Digital S.L.

import logging

def slo_viability(slice_slos, nrp_slos):
    """
    Compare Service Level Objectives (SLOs) between a slice and a Network Resource Partition (NRP).

    This method assesses whether an NRP can satisfy the SLOs of a network slice.

    Args:
        slice_slos (list): Service Level Objectives of the slice
        nrp_slos (dict): Service Level Objectives of the Network Resource Pool

    Returns:
        tuple: A boolean indicating viability and a flexibility score
            - First value: True if NRP meets SLOs, False otherwise
            - Second value: A score representing how well the NRP meets the SLOs
    """
    # Define SLO types for maximum and minimum constraints
    slo_type = {
        "max": ["one-way-delay-maximum", "two-way-delay-maximum", "one-way-delay-percentile", "two-way-delay-percentile",
                "one-way-delay-variation-maximum", "two-way-delay-variation-maximum",
                "one-way-delay-variation-percentile", "two-way-delay-variation-percentile",
                "one-way-packet-loss", "two-way-packet-loss"],
        "min": ["one-way-bandwidth", "two-way-bandwidth", "shared-bandwidth"]
    }
    score = 0
    flexibility_scores = []
    for slo in slice_slos:
        for nrp_slo in nrp_slos['slos']:
            if slo["metric-type"] == nrp_slo["metric-type"]:
                # Handle maximum type SLOs
                if slo["metric-type"] in slo_type["max"]:
                    logging.debug(f"SLO: {slo}, NRP SLO: {nrp_slo}")
                    flexibility = (slo["bound"] - nrp_slo["bound"]) / slo["bound"]
                    if slo["bound"] < nrp_slo["bound"]:
                        return False, 0  # Does not meet maximum constraint
                # Handle minimum type SLOs
                if slo["metric-type"] in slo_type["min"]:
                    logging.debug(f"SLO: {slo}, NRP SLO: {nrp_slo}")
                    flexibility = (nrp_slo["bound"] - slo["bound"]) / slo["bound"]
                    if slo["bound"] > nrp_slo["bound"]:
                        return False, 0  # Does not meet minimum constraint
                flexibility_scores.append(flexibility)
                break  # Exit inner loop after finding matching metric
        
        # Calculate final viability score
        score = sum(flexibility_scores) / len(flexibility_scores) if flexibility_scores else 0
    return True, score  # Si pasó todas las verificaciones, la NRP es viable