# Copyright 2022-2025 ETSI SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)

# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at

#     http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# This file includes original contributions from Telefonica Innovación Digital S.L.

import logging
import time
from src.utils.dump_templates import dump_templates
from src.utils.build_response import build_response
from src.nbi_processor.main import nbi_processor
from src.database.store_data import store_data
from src.mapper.main import mapper
from src.realizer.main import realizer
from src.realizer.send_controller import send_controller

class NSController:
    """
    Network Slice Controller (NSC) - A class to manage network slice creation,
    modification, and deletion across different network domains.

    This controller handles the translation, mapping, and realization of network
    slice intents from different formats (3GPP and IETF) to network-specific
    configurations.

    Key Functionalities:
    - Intent Processing: Translate and process network slice intents
    - Slice Management: Create, modify, and delete network slices
    - NRP (Network Resource Partition) Mapping: Match slice requirements with available resources
    - Slice Realization: Convert intents to specific network configurations (L2VPN, L3VPN)
    """

    def __init__(self, controller_type = "TFS"):
        """
        Initialize the Network Slice Controller.

        Args:
            controller_type (str): Flag to determine if configurations
                should be uploaded to Teraflow or IXIA system.

        Attributes:
            controller_type (str): Flag for Teraflow or Ixia upload
            response (dict): Stores slice creation responses
            start_time (float): Tracks slice setup start time
            end_time (float): Tracks slice setup end time
            setup_time (float): Total time taken for slice setup in milliseconds
        """
        self.controller_type = controller_type

        self.path = ""
        self.response = []
        self.start_time = 0
        self.end_time = 0
        self.setup_time = 0

    def nsc(self, intent_json, slice_id=None):
        """
        Main Network Slice Controller method to process and realize network slice intents.

        Workflow:
        1. Load IETF template
        2. Process intent (detect format, translate if needed)
        3. Extract slice data
        4. Store slice information
        5. Map slice to Network Resource Pool (NRP)
        6. Realize slice configuration
        7. Send configuration to network controllers

        Args:
            intent_json (dict): Network slice intent in 3GPP or IETF format
            slice_id (str, optional): Existing slice identifier for modification

        Returns:
            dict: Contains slice creation responses and setup time in milliseconds

        """
        # Start performance tracking
        self.start_time = time.perf_counter()

        # Reset requests
        requests = {"services":[]}
        response = None

        # Process intent (translate if 3GPP)
        ietf_intents = nbi_processor(intent_json)

        for intent in ietf_intents:
            # Mapper
            rules = mapper(intent)
            # Build response
            self.response = build_response(intent, self.response, controller_type= self.controller_type)
            # Realizer
            request = realizer(intent, controller_type=self.controller_type, response = self.response, rules = rules)
            # Store slice request details
            if request: 
                requests["services"].append(request)
                store_data(intent, slice_id, controller_type=self.controller_type)

        # Store the generated template for debugging
        dump_templates(intent_json, ietf_intents, requests)

        # Check if there are services to process
        if not requests.get("services"):
            raise RuntimeError("No service to process.")

        # Send config to controllers
        response = send_controller(self.controller_type, requests)

        if not response:
            raise Exception("Controller upload failed")

        # End performance tracking
        self.end_time = time.perf_counter()
        setup_time = (self.end_time - self.start_time) * 1000

        return {
            "slices": self.response,
            "setup_time": setup_time
        }