# Copyright 2022-2025 ETSI SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)

# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at

#     http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# This file is an original contribution from Telefonica Innovación Digital S.L.

import logging
from flask import Flask
from flask_restx import Api
from flask_cors import CORS
from swagger.tfs_namespace import tfs_ns
from swagger.ixia_namespace import ixia_ns
from swagger.E2E_namespace import e2e_ns
from src.config.constants import NSC_PORT
from src.webui.gui import gui_bp
from src.config.config import create_config
from src.database.db import init_db

def create_app():
    """Create Flask application with configured API and namespaces."""
    init_db()
    app = Flask(__name__)
    app = create_config(app)
    CORS(app)

    # Configure logging to provide clear and informative log messages
    logging.basicConfig(
        level=app.config["LOGGING_LEVEL"],
        format="%(levelname)s - %(message)s"
    )

    # Create API instance
    api = Api(
        app,
        version="1.0",
        title="Network Slice Controller (NSC) API",
        description="API for orchestrating and realizing transport network slice requests",
        doc="/nsc"  # Swagger UI URL
    )

    # Register namespaces
    api.add_namespace(tfs_ns, path="/tfs")
    api.add_namespace(ixia_ns, path="/ixia")
    api.add_namespace(e2e_ns, path="/e2e")

    if app.config["WEBUI_DEPLOY"]:
        app.secret_key = "clave-secreta-dev"
        app.register_blueprint(gui_bp)

    return app

if __name__ == "__main__":
    app = create_app()
    app.run(host="0.0.0.0", port=NSC_PORT, debug=True)
