# Copyright 2022-2025 ETSI SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)

# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at

#     http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# This file is an original contribution from Telefonica Innovación Digital S.L.

import json, logging, uuid
import requests
import os
import pandas as pd
from flask import render_template, request, jsonify, redirect, url_for, session, Blueprint
from src.config.constants import SRC_PATH, NSC_PORT, TEMPLATES_PATH
from src.realizer.ixia.helpers.NEII_V4 import NEII_controller
from flask import current_app

gui_bp = Blueprint('gui', __name__, template_folder=os.path.join(SRC_PATH, 'webui', 'templates'), static_folder=os.path.join(SRC_PATH, 'webui', 'static'), static_url_path='/webui/static')

#Variables for dev accessing
USERNAME = 'admin'
PASSWORD = 'admin'
enter=False

def __safe_int(value):
    """
    Safely convert a string or numeric input to int or float.
    
    Args:
        value (str|int|float): The input value to convert.
    
    Returns:
        int|float|None: The converted integer or float value, or None if conversion fails.
    """
    try:
        if isinstance(value, str):
            value = value.strip().replace(',', '.')
        number = float(value)
        return int(number) if number.is_integer() else number
    except (ValueError, TypeError, AttributeError):
        return None

def __build_request_ietf(src_node_ip=None, dst_node_ip=None, vlan_id=None, bandwidth=None, latency=None, tolerance=0, latency_version=None, reliability=None):
    """
    Build an IETF-compliant network slice request formm from inputs.
    
    Args: IPs, VLAN, bandwidth, latency, reliability, etc.

    Returns: dict representing the JSON request.
    """
    # Open and read the template file
    with open(os.path.join(TEMPLATES_PATH, 'ietf_template_empty.json'), 'r') as source:
        # Clean up the JSON template
        template = source.read().replace('\t', '').replace('\n', '').replace("'", '"').strip()
    request = json.loads(template)

    request["ietf-network-slice-service:network-slice-services"]["slo-sle-templates"]["slo-sle-template"][0]["id"] = f"qos-profile-{uuid.uuid4()}"

    # Generate unique slice service ID and description
    request["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["id"] = f"slice-service-{uuid.uuid4()}"
    request["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["slo-sle-policy"]["slo-sle-template"] = request["ietf-network-slice-service:network-slice-services"]["slo-sle-templates"]["slo-sle-template"][0]["id"]
    
    # Configure Source SDP
    request["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["sdps"]["sdp"][0]["node-id"] = "source-node" #Pendiente de rellenar
    request["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["sdps"]["sdp"][0]["sdp-ip-address"] = src_node_ip
    request["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["sdps"]["sdp"][0]["service-match-criteria"]["match-criterion"][0]["match-type"] = "VLAN"
    request["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["sdps"]["sdp"][0]["service-match-criteria"]["match-criterion"][0]["value"] = vlan_id
    request["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["sdps"]["sdp"][0]["attachment-circuits"]["attachment-circuit"][0]["ac-ipv4-address"] = "" # Pendiente de rellenar
    request["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["sdps"]["sdp"][0]["attachment-circuits"]["attachment-circuit"][0]["sdp-peering"]["peer-sap-id"] = src_node_ip

    # Configure Destination SDP
    request["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["sdps"]["sdp"][1]["node-id"] = "destination-node"
    request["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["sdps"]["sdp"][1]["sdp-ip-address"] = dst_node_ip
    request["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["sdps"]["sdp"][1]["service-match-criteria"]["match-criterion"][0]["match-type"] = "VLAN"
    request["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["sdps"]["sdp"][1]["service-match-criteria"]["match-criterion"][0]["value"] = vlan_id
    request["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["sdps"]["sdp"][1]["attachment-circuits"]["attachment-circuit"][0]["ac-ipv4-address"] = ""# Pendiente de rellenar
    request["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["sdps"]["sdp"][1]["attachment-circuits"]["attachment-circuit"][0]["sdp-peering"]["peer-sap-id"] = dst_node_ip

    # Configure Connection Group and match-criteria
    request["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["connection-groups"]["connection-group"][0]["id"] = "source-node_destination-node" #Pendiente de rellenar
    request["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["sdps"]["sdp"][0]["service-match-criteria"]["match-criterion"][0]["target-connection-group-id"] = "" #Pendiente de rellenar
    request["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["sdps"]["sdp"][1]["service-match-criteria"]["match-criterion"][0]["target-connection-group-id"] = "" #Pendiente de rellenar

    # Populate template with SLOs (currently supporting QoS profile, latency and bandwidth)
    if bandwidth:
        request["ietf-network-slice-service:network-slice-services"]["slo-sle-templates"]["slo-sle-template"][0]["slo-policy"]["metric-bound"].append({
            "metric-type": "one-way-bandwidth",
            "metric-unit": "Mbps",
            "bound": __safe_int(bandwidth)
        })
    if latency:
        request["ietf-network-slice-service:network-slice-services"]["slo-sle-templates"]["slo-sle-template"][0]["slo-policy"]["metric-bound"].append({
            "metric-type": "one-way-delay-maximum",
            "metric-unit": "milliseconds",
            "bound": __safe_int(latency)
        })
    # Configure gaussian latency or internet if specified
    if latency_version:
        request["ietf-network-slice-service:network-slice-services"]["slo-sle-templates"]["slo-sle-template"][0]["description"] = latency_version
    if tolerance: 
        request["ietf-network-slice-service:network-slice-services"]["slo-sle-templates"]["slo-sle-template"][0]["slo-policy"]["metric-bound"].append( {
        "metric-type": "one-way-delay-variation-maximum",
        "metric-unit": "milliseconds",
        "bound": __safe_int(tolerance)
    })
    if reliability:
        request["ietf-network-slice-service:network-slice-services"]["slo-sle-templates"]["slo-sle-template"][0]["sle-policy"]["reliability"] = __safe_int(reliability) 

    return request

def __build_request(ip_version=None, src_node_ip=None, dst_node_ip=None, src_node_ipv6=None, dst_node_ipv6=None,
                    vlan_id=None, bandwidth=None, latency=None, tolerance=0, latency_version=None,
                    reliability=None, packet_reorder=None, num_pack=None, pack_reorder=None, num_reorder=None,
                    max_reorder=None, desv_reorder=None, drop_version=None, packets_drop=None,
                    drops=None, desv_drop=None):
    """
    Build a JSON request formm from inputs.
    Args: IPs, VLAN, bandwidth, latency, reliability, etc.
    Returns: dict representing the JSON request.
    """
    json_data = {
        "ip_version": ip_version,
        "src_node_ip": src_node_ip,
        "dst_node_ip": dst_node_ip,
        "src_node_ipv6": src_node_ipv6,
        "dst_node_ipv6": dst_node_ipv6,
        "vlan_id": vlan_id,
        "bandwidth": bandwidth,
        "latency": latency,
        "tolerance": tolerance,
        "latency_version": latency_version,
        "reliability": reliability,
        "packet_reorder": packet_reorder,
        "num_pack": num_pack,
        "pack_reorder": pack_reorder,
        "num_reorder": num_reorder,
        "max_reorder": max_reorder,
        "desv_reorder": desv_reorder,
        "drop_version": drop_version,
        "packets_drop": packets_drop,
        "drops": drops,
        "desv_drop": desv_drop
    }
    return json_data

def __datos_json():
    """
    Read slice data from JSON file and return as a pandas DataFrame.
    Returns:
        pd.DataFrame: DataFrame containing slice data.
    """
    try:
        with open(os.path.join(SRC_PATH, 'slice_ddbb.json'), 'r') as fichero:
            datos =json.load(fichero)
            rows =[]
            for source_ip, source_info in datos["source"].items():
                vlan = source_info["vlan"]
                for dest_ip, dest_info in source_info["destination"].items():
                    row ={
                        "Source IP": source_ip,
                        "Destiny IP": dest_ip,
                        "VLAN": vlan,
                        "attributes": dest_info["attributes"]
                    }
                    rows.append(row)
            dataframe =pd.DataFrame(rows)
    except FileNotFoundError:
        dataframe =pd.DataFrame()
    except ValueError as e:
        dataframe =pd.DataFrame()
    return dataframe

@gui_bp.route('/webui')
def home():
    session['enter'] = False
    # Leer las IPs actuales del archivo de configuración
    try:
        tfs_ip = current_app.config["TFS_IP"]
        ixia_ip = current_app.config["IXIA_IP"]
    except Exception:
        tfs_ip = 'No configurada'
        ixia_ip = 'No configurada'
    return render_template('welcome.html', tfs_ip=tfs_ip, ixia_ip=ixia_ip)

@gui_bp.route('/webui/generate/tfs', methods=['POST','GET'])
def generate_tfs():
    session['enter']=False
    if request.method =='POST':
        src_node_ip =request.form['src_node_ip']
        dst_node_ip =request.form['dst_node_ip']
        vlan_id =request.form['vlan_id']
        if vlan_id == '':
            vlan_id = None
        latency =request.form['latency_intent']
        bandwidth =request.form['bandwidth_intent']

        slice_request = __build_request_ietf(src_node_ip=src_node_ip, dst_node_ip=dst_node_ip, vlan_id=vlan_id, latency=latency, bandwidth=bandwidth)

        # Si 'request' es un diccionario, conviértelo a JSON
        json_data = json.dumps(slice_request)
        files = {
            'file': ('ietf_template_example.json', json_data, 'application/json')
        }

        response = requests.post(
            f'http://localhost:{NSC_PORT}/tfs/slice',
            headers={
                'accept': 'application/json'
            },
            files=files
        )
        response.raise_for_status()

        if response.ok:
            return redirect(url_for('gui.generate_tfs', src_node_ip=src_node_ip, dst_node_ip=dst_node_ip, vlan_id=vlan_id, latency=latency, bandwidth=bandwidth))
        else:
            return render_template('index.html', error="Error al generar el intent")

    src_node_ip =request.args.get('src_node_ip', '')
    dst_node_ip =request.args.get('dst_node_ip', '')
    vlan_id =request.args.get('vlan_id', '')
    latency =request.args.get('latency', '')
    bandwidth =request.args.get('bandwidth', '')
    reliability=request.args.get('reliability', '')

    return render_template('index.html', src_node_ip=src_node_ip, dst_node_ip=dst_node_ip, vlan_id=vlan_id, latency=latency, bandwidth=bandwidth, reliability=reliability)

@gui_bp.route('/webui/generate/ixia', methods=['GET','POST'])
def generate_ixia():
    session['enter']=False
    if request.method =='POST':
        src_node_ip =request.form['src_node_ip']
        dst_node_ip =request.form['dst_node_ip']
        vlan_id =request.form['vlan_id']
        latency =request.form['latency_intent']
        bandwidth =request.form['bandwidth_intent']
        latency_version="internet"
        tolerance= request.form['tolerance_intent']
        reliability=request.form['reliability']

        if int(reliability)==100: reliability=None
        slice_request = __build_request_ietf(src_node_ip=src_node_ip, dst_node_ip=dst_node_ip, vlan_id=vlan_id, bandwidth=bandwidth, latency=latency, latency_version=latency_version, tolerance=tolerance, reliability=reliability)
        # Si 'request' es un diccionario, conviértelo a JSON
        json_data = json.dumps(slice_request)
        files = {
            'file': ('ietf_template_example.json', json_data, 'application/json')
        }

        try:
            response = requests.post(
                f'http://localhost:{NSC_PORT}/ixia/slice',
                headers={'accept': 'application/json'},
                files=files
            )
            response.raise_for_status()
            if response.ok:
                return redirect(url_for('gui.generate_ixia', src_node_ip=src_node_ip, dst_node_ip=dst_node_ip, vlan_id=vlan_id, latency=latency, tolerance=tolerance,bandwidth=bandwidth,reliability=reliability))
        except requests.RequestException as e:
            print("HTTP error:", e)
            return render_template('ixia.html', error="Intent Generation Error")

    src_node_ip =request.args.get('src_node_ip', '')
    dst_node_ip =request.args.get('dst_node_ip', '')
    vlan_id =request.args.get('vlan_id', '')
    latency =request.args.get('latency', '')
    tolerance=request.args.get('tolerance', '')
    bandwidth =request.args.get('bandwidth', '')
    reliability=request.args.get('reliability','')

    return render_template('ixia.html', src_node_ip=src_node_ip, dst_node_ip=dst_node_ip, vlan_id=vlan_id, latency=latency, tolerance=tolerance, bandwidth=bandwidth, reliability=reliability)


@gui_bp.route('/webui/dev', methods=['GET', 'POST'])
def develop():
    print('Session content:', dict(session))
    if 'enter' not in session or session['enter'] is False:
        return redirect(url_for('gui.login'))

    if request.method == 'POST':
        ip_version = request.form.get('ip_version', '')
        src_node_ipv4 = request.form.get('src_node_ipv4', '')
        dst_node_ipv4 = request.form.get('dst_node_ipv4', '')
        src_node_ipv6 = request.form.get('src_node_ipv6', '')
        dst_node_ipv6 = request.form.get('dst_node_ipv6', '')
        vlan_id = request.form.get('vlan_id', '')
        bandwidth = request.form.get('bandwidth_intent', '')
        latency = request.form.get('latency_intent', '')
        latency_version = request.form.get('delay_statistic', '')
        tolerance = request.form.get('max_delay', '')
        packet_reorder = request.form.get('packet_options', '')
        num_pack = request.form.get('numPack', '')
        pack_reorder = request.form.get('packReod', '')
        num_reorder = request.form.get('numReod', '')
        max_reorder = request.form.get('num_maxReod', '')
        desv_reorder = request.form.get('desv_reord', '')
        drop_version = request.form.get('object_options', '')
        packets_drop = request.form.get('packets', '')
        drops = request.form.get('drop', '')
        desv_drop = request.form.get('desv_drop', '')

        json_data = __build_request(ip_version=ip_version, src_node_ip=src_node_ipv4, dst_node_ip=dst_node_ipv4, src_node_ipv6=src_node_ipv6, dst_node_ipv6=dst_node_ipv6, vlan_id=vlan_id, latency=latency, bandwidth=bandwidth, latency_version=latency_version, tolerance=tolerance, packet_reorder=packet_reorder, num_pack=num_pack, pack_reorder=pack_reorder, num_reorder=num_reorder, max_reorder=max_reorder, desv_reorder=desv_reorder, drop_version=drop_version, packets_drop=packets_drop, drops=drops, desv_drop=desv_drop)
        logging.debug("Generated JSON data: %s", json_data)
        if not current_app.config["DUMMY_MODE"]:
            NEII_controller().nscNEII(json_data)

        session['enter'] = True
        return render_template('dev.html',
            json_data=json_data,
            src_node_ip=src_node_ipv4 or src_node_ipv6,
            dst_node_ip=dst_node_ipv4 or dst_node_ipv6,
            vlan_id=vlan_id,
            latency=latency,
            bandwidth=bandwidth,
            latency_version=latency_version,
            tolerance=tolerance,
            packet_reorder=packet_reorder,
            num_pack=num_pack,
            pack_reorder=pack_reorder,
            num_reorder=num_reorder,
            max_reorder=max_reorder,
            desv_reorder=desv_reorder,
            packets_drop=packets_drop,
            drops=drops,
            desv_drop=desv_drop
        )

    return render_template('dev.html',
        src_node_ip=request.args.get('src_node_ip', ''),
        dst_node_ip=request.args.get('dst_node_ip', ''),
        vlan_id=request.args.get('vlan_id', ''),
        latency=request.args.get('latency', ''),
        bandwidth=request.args.get('bandwidth', ''),
        latency_version=request.args.get('latency_version', ''),
        tolerance=request.args.get('max_latency', ''),
        packet_reorder=request.args.get('packet_reorder', ''),
        num_pack=request.args.get('num_pack', ''),
        pack_reorder=request.args.get('pack_reorder', ''),
        num_reorder=request.args.get('num_reorder', ''),
        max_reorder=request.args.get('max_reorder', ''),
        desv_reorder=request.args.get('desv', ''),
        packets_drop=request.args.get('packets_drop', ''),
        drops=request.args.get('drop', ''),
        desv_drop=request.args.get('desv_drop', ''),
        json_data=None
    )

@gui_bp.route('/webui/search', methods=['GET', 'POST'])
def search():

    session['enter'] = False
    
    try:
        response = requests.get(f"http://localhost:{NSC_PORT}/tfs/slice", headers={"accept": "application/json"})
        response.raise_for_status()
        tfs_slices = response.json()

        response = requests.get(f"http://localhost:{NSC_PORT}/ixia/slice", headers={"accept": "application/json"})
        response.raise_for_status()
        ixia_slices = response.json()

        # Combine slices from both controllers
        slices = tfs_slices + ixia_slices
       
    except requests.RequestException as e:
        logging.error("Error fetching slices: %s", e)
        return render_template('search.html', error="No se pudieron obtener los slices.", dataframe_html="")

    # Extract relevant data for DataFrame
    rows = []
    for item in slices:
        try:
            slice_service = item["intent"]["ietf-network-slice-service:network-slice-services"]["slice-service"][0]
            sdp = slice_service["sdps"]["sdp"]
            metric_bound = item["intent"]["ietf-network-slice-service:network-slice-services"]["slo-sle-templates"]["slo-sle-template"][0]["slo-policy"]["metric-bound"]

            source_ip = sdp[0]["sdp-ip-address"]
            dest_ip = sdp[1]["sdp-ip-address"]
            vlan = sdp[0]["service-match-criteria"]["match-criterion"][0]["value"]
            controller = item["controller"]

            # Build attributes list
            attributes = []
            for metric in metric_bound:
                if metric.get("metric-type", "") == "one-way-bandwidth":
                    metric_type = "bandwidth"
                elif metric.get("metric-type", "") == "one-way-delay-maximum":
                    metric_type = "latency"
                elif metric.get("metric-type", "") == "one-way-delay-variation-maximum":
                    metric_type = "tolerance"
                bound = metric.get("bound")
                unit = metric.get("metric-unit", "")
                if bound is not None:
                    attributes.append(f"{metric_type}: {bound} {unit}")

            rows.append({
                "Source IP": source_ip,
                "Destiny IP": dest_ip,
                "Controller": controller,
                "VLAN": vlan,
                "attributes": attributes
            })
        except Exception as e:
            print(f"Error procesando slice: {e}")


    import pandas as pd
    dataframe = pd.DataFrame(rows)

    def format_attributes(attributes):
        formatted_attrs = []
        for attr in attributes:
            formatted_attrs.append(attr)
        if formatted_attrs:
            return formatted_attrs
        else:
            return None

    dataframe['attributes'] = dataframe['attributes'].apply([format_attributes])

    if request.method == 'POST':
        search_option = request.form.get('search_option')
        search_value = request.form.get('search_value')
        if search_option == 'Source IP':
            results = dataframe[dataframe['Source IP'] == search_value]
        elif search_option == 'Destiny IP':
            results = dataframe[dataframe['Destiny IP'] == search_value]
        elif search_option == 'Controller':
            results = dataframe[dataframe['Controller'] == search_value]
        elif search_option == 'VLAN':
            results = dataframe[dataframe['VLAN'] == search_value]
        else:
            results = dataframe
        result_html = results.to_html(classes='table table-striped')
        return jsonify({'result': result_html})

    dataframe_html = dataframe.to_html(classes='table table-striped')
    return render_template('search.html', dataframe_html=dataframe_html)

@gui_bp.route('/webui/login', methods=['GET', 'POST'])
def login():
    global enter
    if request.method == 'POST':
        username = request.form['username']
        password = request.form['password']
        if username == USERNAME and password == PASSWORD:
            session['enter']=True
            return redirect(url_for('gui.develop'))
        else:
            return render_template('login.html', error="Credenciales incorrectas")
    
    return render_template('login.html')

@gui_bp.route('/webui/reset', methods=['POST'])
def reset():
    dataframe=__datos_json()
    dataframe_html =dataframe.to_html(classes='table table-striped')
    return jsonify({'result': dataframe_html})

@gui_bp.route('/webui/update_ips', methods=['POST'])
def update_ips():
    data = request.get_json()
    tfs_ip = data.get('tfs_ip')
    ixia_ip = data.get('ixia_ip')

    # Load existing IPs from the configuration file
    config_path = os.path.join(SRC_PATH, 'IPs.json')
    if os.path.exists(config_path):
        with open(config_path) as f:
            ips = json.load(f)
        ips = {
            "TFS_IP": current_app.config["TFS_IP"],
            "IXIA_IP": current_app.config["IXIA_IP"]
        }
    else:
        ips = {"TFS_IP": "", "IXIA_IP": ""}

    # Update IPs if provided
    if tfs_ip:
        ips['TFS_IP'] = tfs_ip
    if ixia_ip:
        ips['IXIA_IP'] = ixia_ip

    # Save updated IPs back to the file
    with open(config_path, 'w') as f:
        json.dump(ips, f, indent=4)

    return '', 200