# Copyright 2022-2025 ETSI SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)

# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at

#     http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# This file is an original contribution from Telefonica Innovación Digital S.L.

from .safe_get import safe_get

def build_response(intent, response, controller_type = None):
    """
    Build a structured response from network slice intent.
    
    Extracts key information from an IETF network slice intent and formats it
    into a standardized response structure with slice details and QoS requirements.

    Args:
        intent (dict): IETF network slice service intent containing:
            - slice-service: Service configuration with SDPs and IDs
            - slo-sle-templates: QoS policy templates
        response (list): Existing response list to append to
        controller_type (str, optional): Type of controller managing the slice.
                                        Defaults to None

    Returns:
        list: Updated response list with appended slice information containing:
            - id: Slice service identifier
            - source: Source service delivery point ID
            - destination: Destination service delivery point ID
            - vlan: VLAN identifier from match criteria
            - requirements: List of QoS constraint dictionaries with:
                * constraint_type: Metric type and unit (e.g., "latency[ms]")
                * constraint_value: Bound value as string
                
    Notes:
        - Extracts metric bounds from SLO policy (bandwidth, delay, jitter, etc.)
        - Includes availability and MTU if specified in SLO policy
        - Assumes point-to-point topology with exactly 2 SDPs
        - VLAN extracted from first SDP's first match criterion
    """
    
    id = safe_get(intent, ["ietf-network-slice-service:network-slice-services","slice-service",0,"id"])
    source = safe_get(intent, ["ietf-network-slice-service:network-slice-services","slice-service",0,"sdps","sdp",0,"id"])
    destination = safe_get(intent, ["ietf-network-slice-service:network-slice-services","slice-service",0,"sdps","sdp",1,"id"])
    vlan = safe_get(intent, ["ietf-network-slice-service:network-slice-services","slice-service",0,"sdps","sdp",0,"service-match-criteria","match-criterion",0,"value"])

    qos_requirements = []

    # Populate response with QoS requirements and VLAN from intent
    slo_policy = safe_get(intent, ["ietf-network-slice-service:network-slice-services","slo-sle-templates","slo-sle-template",0,"slo-policy"])

    # Process metrics
    for metric in slo_policy.get("metric-bound", []):
        constraint_type = f"{metric['metric-type']}[{metric['metric-unit']}]"
        constraint_value = str(metric["bound"])
        qos_requirements.append({
            "constraint_type": constraint_type,
            "constraint_value": constraint_value
        })

    # Availability
    if "availability" in slo_policy:
        qos_requirements.append({
            "constraint_type": "availability[%]",
            "constraint_value": str(slo_policy["availability"])
        })

    # MTU
    if "mtu" in slo_policy:
        qos_requirements.append({
            "constraint_type": "mtu[bytes]",
            "constraint_value": str(slo_policy["mtu"])
        })
    response.append({
        "id": id,
        "source": source,
        "destination": destination,
        "vlan": vlan,
        "requirements": qos_requirements,
    })
    return response