import json
import pytest
import os

from src.utils.load_template import load_template
from src.utils.dump_templates import dump_templates
from src.utils.send_response import send_response
from src.utils.build_response import build_response
from flask import Flask

@pytest.fixture
def tmp_json_file(tmp_path):
    """Crea un archivo JSON temporal válido y devuelve su ruta y contenido."""
    data = {"name": "test"}
    file_path = tmp_path / "template.json"
    file_path.write_text(json.dumps(data))
    return file_path, data


def test_load_template_ok(tmp_json_file):
    """Debe cargar correctamente un JSON válido."""
    file_path, expected = tmp_json_file
    result = load_template(str(file_path))
    assert result == expected


def test_load_template_invalid(tmp_path):
    """Debe devolver un response con error si el JSON es inválido."""
    bad_file = tmp_path / "bad.json"
    bad_file.write_text("{invalid json}")

    result, code = load_template(str(bad_file))
    assert code == 500
    assert result["success"] is False
    assert "Template loading error" in result["error"]

def test_dump_templates_enabled(monkeypatch, tmp_path):
    """Debe volcar múltiples JSON correctamente en src/templates si DUMP_TEMPLATES está activado."""
    templates_dir = tmp_path / "src" / "templates"
    templates_dir.mkdir(parents=True)

    monkeypatch.setattr("src.utils.dump_templates.TEMPLATES_PATH", str(templates_dir))

    app = Flask(__name__)
    app.config["DUMP_TEMPLATES"] = True

    with app.app_context():
        nbi = {"nbi": 1}
        ietf = {"ietf": 2}
        realizer = {"realizer": 3}

        dump_templates(nbi, ietf, realizer)

    for name, data in [("nbi_template.json", nbi), ("ietf_template.json", ietf), ("realizer_template.json", realizer)]:
        file_path = templates_dir / name
        assert file_path.exists()
        assert json.loads(file_path.read_text()) == data

def test_dump_templates_disabled(monkeypatch, tmp_path):
    """No debe escribir nada en src/templates si DUMP_TEMPLATES está desactivado."""
    templates_dir = tmp_path / "src" / "templates"
    templates_dir.mkdir(parents=True)

    monkeypatch.setattr("src.utils.dump_templates.TEMPLATES_PATH", str(templates_dir))

    app = Flask(__name__)
    app.config["DUMP_TEMPLATES"] = False

    with app.app_context():
        dump_templates({"nbi": 1}, {"ietf": 2}, {"realizer": 3})

    for name in ["nbi_template.json", "ietf_template.json", "realizer_template.json"]:
        assert not (templates_dir / name).exists()

def test_send_response_success():
    """Debe devolver success=True y code=200 si el resultado es True."""
    resp, code = send_response(True, data={"k": "v"})
    assert code == 200
    assert resp["success"] is True
    assert resp["data"]["k"] == "v"
    assert resp["error"] is None


def test_send_response_error():
    """Debe devolver success=False y code=400 si el resultado es False."""
    resp, code = send_response(False, message="fallo")
    assert code == 400
    assert resp["success"] is False
    assert resp["data"] is None
    assert "fallo" in resp["error"]

def ietf_intent():
    """Intento válido en formato IETF simplificado."""
    return {
        "ietf-network-slice-service:network-slice-services": {
            "slo-sle-templates": {
                "slo-sle-template": [
                    {
                        "id": "qos1",
                        "slo-policy": {
                            "metric-bound": [
                                {
                                    "metric-type": "one-way-bandwidth",
                                    "metric-unit": "kbps",
                                    "bound": 1000
                                }
                            ],
                            "availability": 99.9,
                            "mtu": 1500
                        }
                    }
                ]
            },
            "slice-service": [
                {
                    "id": "slice-test-1",
                    "sdps": {
                        "sdp": [
                            {
                                "id": "CU",
                                "sdp-ip-address": "10.0.0.1",
                                "service-match-criteria": {
                                    "match-criterion": [{"match-type": "vlan", "value": "100"}]
                                },
                            },
                            {
                                "id": "DU",
                                "sdp-ip-address": "10.0.0.2",
                                "service-match-criteria": {
                                    "match-criterion": [{"match-type": "vlan", "value": "100"}]
                                },
                            },
                        ]
                    },
                }
            ],
        }
    }


def test_build_response_ok():
    """Debe construir correctamente el response a partir de un intent IETF válido."""
    intent = ietf_intent()
    response = []
    result = build_response(intent, response)

    assert isinstance(result, list)
    assert len(result) == 1

    slice_data = result[0]
    assert slice_data["id"] == "slice-test-1"
    assert slice_data["source"] == "CU"
    assert slice_data["destination"] == "DU"
    assert slice_data["vlan"] == "100"

    # Validar constraints
    requirements = slice_data["requirements"]
    assert any(r["constraint_type"] == "one-way-bandwidth[kbps]" and r["constraint_value"] == "1000" for r in requirements)
    assert any(r["constraint_type"] == "availability[%]" and r["constraint_value"] == "99.9" for r in requirements)
    assert any(r["constraint_type"] == "mtu[bytes]" and r["constraint_value"] == "1500" for r in requirements)


def test_build_response_empty_policy():
    """Debe devolver lista sin constraints si slo-policy está vacío."""
    intent = ietf_intent()
    intent["ietf-network-slice-service:network-slice-services"]["slo-sle-templates"]["slo-sle-template"][0]["slo-policy"] = {}
    response = []
    result = build_response(intent, response)

    assert isinstance(result, list)
    assert len(result[0]["requirements"]) == 0


def test_build_response_invalid_intent():
    """Debe fallar limpiamente si el intent no tiene la estructura esperada."""
    bad_intent = {}
    response = []
    try:
        result = build_response(bad_intent, response)
    except Exception:
        result = []
    assert result == []
