import pytest
from unittest.mock import patch
from src.nbi_processor.detect_format import detect_format
from src.nbi_processor.main import nbi_processor
from src.nbi_processor.translator import translator


# ---------- Tests detect_format ----------

def test_detect_format_ietf():
    data = {"ietf-network-slice-service:network-slice-services": {}}
    assert detect_format(data) == "IETF"

def test_detect_format_3gpp_variants():
    assert detect_format({"RANSliceSubnet1": {}}) == "3GPP"
    assert detect_format({"NetworkSlice1": {}}) == "3GPP"
    assert detect_format({"TopSliceSubnet1": {}}) == "3GPP"
    assert detect_format({"CNSliceSubnet1": {}}) == "3GPP"

def test_detect_format_none():
    assert detect_format({"foo": "bar"}) is None


# ---------- Fixtures ----------

@pytest.fixture
def ietf_intent():
    return {"ietf-network-slice-service:network-slice-services": {"foo": "bar"}}

@pytest.fixture
def gpp_intent():
    # Estructura mínima consistente con translator
    return {
        "RANSliceSubnet1": {
            "networkSliceSubnetRef": ["subnetA", "subnetB"]
        },
        "subnetA": {
            "EpTransport": ["EpTransport ep1", "EpTransport ep2"],
            "SliceProfileList": [{
                "RANSliceSubnetProfile": {
                    "dLThptPerSliceSubnet": {
                        "GuaThpt": 1,
                        "MaxThpt": 2
                    },
                    "uLThptPerSliceSubnet": {
                        "GuaThpt": 1,
                        "MaxThpt": 2
                    },
                    "dLLatency": 20,
                    "uLLatency": 20
                }
            }],
        },
        "subnetB": {
            "EpTransport": ["EpTransport ep3", "EpTransport ep4"],
        },
        "EpTransport ep1": {
            "qosProfile": "qosA",
            "EpApplicationRef": ["EP_N2 epRef1"],
            "logicalInterfaceInfo": {"logicalInterfaceType": "typeA", "logicalInterfaceId": "idA"},
            "IpAddress": "1.1.1.1",
            "NextHopInfo": "NH1",
        },
        "EpTransport ep2": {
            "qosProfile": "qosB",
            "EpApplicationRef": ["EP_N2 epRef2"],
            "logicalInterfaceInfo": {"logicalInterfaceType": "typeB", "logicalInterfaceId": "idB"},
            "IpAddress": "2.2.2.2",
            "NextHopInfo": "NH2",
        },
        "EP_N2 epRef1": {"localAddress": "10.0.0.1", "remoteAddress": "11.1.1.1", "epTransportRef": "ep1"},
        "EP_N2 epRef2": {"localAddress": "10.0.0.2", "remoteAddress": "11.1.1.2", "epTransportRef": "ep2"},
        "EpTransport ep3": {"qosProfile": "qosC", "EpApplicationRef": ["EP_N2 epRef3"], "logicalInterfaceInfo": {"logicalInterfaceType": "typeC", "logicalInterfaceId": "idC"}, "IpAddress": "3.3.3.3", "NextHopInfo": "NH3"},
        "EpTransport ep4": {"qosProfile": "qosD", "EpApplicationRef": ["EP_N2 epRef4"], "logicalInterfaceInfo": {"logicalInterfaceType": "typeD", "logicalInterfaceId": "idD"}, "IpAddress": "4.4.4.4", "NextHopInfo": "NH4"},
        "EP_N2 epRef3": {"localAddress": "10.0.0.3", "remoteAddress": "11.1.1.3", "epTransportRef": "ep3"},
        "EP_N2 epRef4": {"localAddress": "10.0.0.4", "remoteAddress": "11.1.1.4", "epTransportRef": "ep4"},
    }


@pytest.fixture
def fake_template():
    # Plantilla mínima para que el traductor funcione
    return {
        "ietf-network-slice-service:network-slice-services": {
            "slo-sle-templates": {
                "slo-sle-template": [
                    {"id": "", "slo-policy": {"metric-bound": []}}
                ]
            },
            "slice-service": [
                {
                    "id": "",
                    "description": "",
                    "slo-sle-policy": {},
                    "sdps": {"sdp": [
                        {"service-match-criteria": {"match-criterion": [{}]}, "attachment-circuits": {"attachment-circuit": [{"sdp-peering": {}}]}},
                        {"service-match-criteria": {"match-criterion": [{}]}, "attachment-circuits": {"attachment-circuit": [{"sdp-peering": {}}]}}
                    ]},
                    "connection-groups": {"connection-group": [{}]},
                }
            ],
        }
    }


# ---------- Tests nbi_processor ----------

def test_nbi_processor_ietf(ietf_intent):
    result = nbi_processor(ietf_intent)
    assert isinstance(result, list)
    assert result[0] == ietf_intent

@patch("src.nbi_processor.main.translator")
def test_nbi_processor_3gpp(mock_translator, gpp_intent):
    mock_translator.return_value = {"ietf-network-slice-service:network-slice-services": {}}
    result = nbi_processor(gpp_intent)
    assert isinstance(result, list)
    assert len(result) == 2  # Dos subnets procesados
    assert all("ietf-network-slice-service:network-slice-services" in r for r in result)

def test_nbi_processor_unrecognized():
    with pytest.raises(ValueError):
        nbi_processor({"foo": "bar"})

def test_nbi_processor_empty():
    with pytest.raises(ValueError):
        nbi_processor({})


# ---------- Tests translator ----------

@patch("src.nbi_processor.translator.load_template")
def test_translator_basic(mock_load_template, gpp_intent, fake_template):
    mock_load_template.return_value = fake_template
    result = translator(gpp_intent, "subnetA")

    assert isinstance(result, dict)
    assert "ietf-network-slice-service:network-slice-services" in result

    slice_service = result["ietf-network-slice-service:network-slice-services"]["slice-service"][0]
    assert slice_service["id"].startswith("slice-service-")
    assert "description" in slice_service
    assert slice_service["slo-sle-policy"]["slo-sle-template"] == "qosA"  # viene del ep1

import re
import uuid


# ---------- Extra detect_format ----------

@pytest.mark.parametrize("data", [
    None,
    [],
    "",
    123,
])
def test_detect_format_invalid_types(data):
    assert detect_format(data if isinstance(data, dict) else {}) in (None, "IETF", "3GPP")


def test_detect_format_multiple_keys():
    # Si tiene IETF y 3GPP, debe priorizar IETF
    data = {
        "ietf-network-slice-service:network-slice-services": {},
        "RANSliceSubnet1": {}
    }
    assert detect_format(data) == "IETF"


# ---------- Extra nbi_processor ----------

def test_nbi_processor_gpp_missing_refs(gpp_intent):
    # Quitar networkSliceSubnetRef debería provocar ValueError en translator loop
    broken = gpp_intent.copy()
    broken["RANSliceSubnet1"] = {}  # no tiene "networkSliceSubnetRef"
    with pytest.raises(KeyError):
        nbi_processor(broken)


# ---------- Extra translator ----------

@patch("src.nbi_processor.translator.load_template")
def test_translator_maps_metrics(mock_load_template, gpp_intent, fake_template):
    mock_load_template.return_value = fake_template
    result = translator(gpp_intent, "subnetA")

    metrics = result["ietf-network-slice-service:network-slice-services"]["slo-sle-templates"]["slo-sle-template"][0]["slo-policy"]["metric-bound"]
    metric_types = {m["metric-type"] for m in metrics}
    assert "one-way-delay-maximum" in metric_types
    assert "one-way-bandwidth" in metric_types


@patch("src.nbi_processor.translator.load_template")
def test_translator_empty_profile(mock_load_template, gpp_intent, fake_template):
    mock_load_template.return_value = fake_template
    gpp_intent["subnetA"]["SliceProfileList"] = [{}]  # vacío
    result = translator(gpp_intent, "subnetA")
    metrics = result["ietf-network-slice-service:network-slice-services"]["slo-sle-templates"]["slo-sle-template"][0]["slo-policy"]["metric-bound"]
    assert metrics == []  # no debería añadir nada

@patch("src.nbi_processor.translator.load_template")
def test_translator_sdps_are_populated(mock_load_template, gpp_intent, fake_template):
    mock_load_template.return_value = fake_template
    result = translator(gpp_intent, "subnetA")
    slice_service = result["ietf-network-slice-service:network-slice-services"]["slice-service"][0]

    sdp0 = slice_service["sdps"]["sdp"][0]
    assert sdp0["node-id"] == "ep1"
    assert re.match(r"^\d+\.\d+\.\d+\.\d+$", sdp0["attachment-circuits"]["attachment-circuit"][0]["ac-ipv4-address"])
    assert "target-connection-group-id" in sdp0["service-match-criteria"]["match-criterion"][0]

    sdp1 = slice_service["sdps"]["sdp"][1]
    assert sdp1["node-id"] == "ep2"
    assert sdp1["attachment-circuits"]["attachment-circuit"][0]["sdp-peering"]["peer-sap-id"].startswith("NH")


@patch("src.nbi_processor.translator.load_template")
def test_translator_with_single_endpoint_should_fail(mock_load_template, gpp_intent, fake_template):
    mock_load_template.return_value = fake_template
    gpp_intent["subnetA"]["EpTransport"] = ["EpTransport ep1"]  # solo uno
    with pytest.raises(IndexError):
        translator(gpp_intent, "subnetA")
