import pytest
import logging
from unittest.mock import patch, MagicMock, call
from flask import Flask
from src.mapper.main import mapper
from src.mapper.slo_viability import slo_viability


@pytest.fixture
def sample_ietf_intent():
    """Fixture providing sample IETF network slice intent."""
    return {
        "ietf-network-slice-service:network-slice-services": {
            "slice-service": [{
                "id": "slice-service-12345",
                "description": "Test network slice",
                "service-tags": {"tag-type": {"value": "L2VPN"}}
            }],
            "slo-sle-templates": {
                "slo-sle-template": [{
                    "id": "profile1",
                    "slo-policy": {
                        "metric-bound": [
                            {
                                "metric-type": "one-way-bandwidth",
                                "metric-unit": "kbps",
                                "bound": 1000
                            },
                            {
                                "metric-type": "one-way-delay-maximum",
                                "metric-unit": "milliseconds",
                                "bound": 10
                            }
                        ]
                    }
                }]
            }
        }
    }


@pytest.fixture
def sample_nrp_view():
    """Fixture providing sample NRP view."""
    return [
        {
            "id": "nrp-1",
            "available": True,
            "slices": [],
            "slos": [
                {
                    "metric-type": "one-way-bandwidth",
                    "bound": 1500
                },
                {
                    "metric-type": "one-way-delay-maximum",
                    "bound": 8
                }
            ]
        },
        {
            "id": "nrp-2",
            "available": True,
            "slices": [],
            "slos": [
                {
                    "metric-type": "one-way-bandwidth",
                    "bound": 500
                },
                {
                    "metric-type": "one-way-delay-maximum",
                    "bound": 15
                }
            ]
        },
        {
            "id": "nrp-3",
            "available": False,
            "slices": [],
            "slos": [
                {
                    "metric-type": "one-way-bandwidth",
                    "bound": 2000
                },
                {
                    "metric-type": "one-way-delay-maximum",
                    "bound": 5
                }
            ]
        }
    ]


@pytest.fixture
def mock_app():
    """Fixture providing mock Flask app context."""
    app = Flask(__name__)
    app.config = {
        "NRP_ENABLED": False,
        "PLANNER_ENABLED": False,
        "SERVER_NAME": "localhost",
        "APPLICATION_ROOT": "/",
        "PREFERRED_URL_SCHEME": "http"  
    }
    return app


@pytest.fixture
def app_context(mock_app):
    """Fixture providing Flask application context."""
    with mock_app.app_context():
        yield mock_app


class TestSloViability:
    """Tests for slo_viability function."""
    
    def test_slo_viability_meets_all_requirements(self):
        """Test when NRP meets all SLO requirements."""
        slice_slos = [
            {
                "metric-type": "one-way-bandwidth",
                "bound": 1000
            },
            {
                "metric-type": "one-way-delay-maximum",
                "bound": 10
            }
        ]
        
        nrp_slos = {
            "slos": [
                {
                    "metric-type": "one-way-bandwidth",
                    "bound": 1500
                },
                {
                    "metric-type": "one-way-delay-maximum",
                    "bound": 8
                }
            ]
        }
        
        viable, score = slo_viability(slice_slos, nrp_slos)
        
        assert viable is True
        assert score > 0
    
    def test_slo_viability_fails_bandwidth_minimum(self):
        """Test when NRP doesn't meet minimum bandwidth requirement."""
        slice_slos = [
            {
                "metric-type": "one-way-bandwidth",
                "bound": 1000
            }
        ]
        
        nrp_slos = {
            "slos": [
                {
                    "metric-type": "one-way-bandwidth",
                    "bound": 500  # Less than required
                }
            ]
        }
        
        viable, score = slo_viability(slice_slos, nrp_slos)
        
        assert viable is False
        assert score == 0
    
    def test_slo_viability_fails_delay_maximum(self):
        """Test when NRP doesn't meet maximum delay requirement."""
        slice_slos = [
            {
                "metric-type": "one-way-delay-maximum",
                "bound": 10
            }
        ]
        
        nrp_slos = {
            "slos": [
                {
                    "metric-type": "one-way-delay-maximum",
                    "bound": 15  # Greater than maximum allowed
                }
            ]
        }
        
        viable, score = slo_viability(slice_slos, nrp_slos)
        
        assert viable is False
        assert score == 0
    
    def test_slo_viability_multiple_metrics_partial_failure(self):
        """Test when one metric fails in a multi-metric comparison."""
        slice_slos = [
            {
                "metric-type": "one-way-bandwidth",
                "bound": 1000
            },
            {
                "metric-type": "one-way-delay-maximum",
                "bound": 10
            }
        ]
        
        nrp_slos = {
            "slos": [
                {
                    "metric-type": "one-way-bandwidth",
                    "bound": 1500  # OK
                },
                {
                    "metric-type": "one-way-delay-maximum",
                    "bound": 15  # NOT OK
                }
            ]
        }
        
        viable, score = slo_viability(slice_slos, nrp_slos)
        
        assert viable is False
        assert score == 0
    
    def test_slo_viability_flexibility_score_calculation(self):
        """Test flexibility score calculation."""
        slice_slos = [
            {
                "metric-type": "one-way-bandwidth",
                "bound": 1000
            }
        ]
        
        nrp_slos = {
            "slos": [
                {
                    "metric-type": "one-way-bandwidth",
                    "bound": 2000  # 100% better than requirement
                }
            ]
        }
        
        viable, score = slo_viability(slice_slos, nrp_slos)
        
        assert viable is True
        # Flexibility = (2000 - 1000) / 1000 = 1.0
        assert score == 1.0
    
    def test_slo_viability_empty_slos(self):
        """Test with empty SLO list."""
        slice_slos = []
        nrp_slos = {"slos": []}
        
        viable, score = slo_viability(slice_slos, nrp_slos)
        
        assert viable is True
        assert score == 0
    
    def test_slo_viability_no_matching_metrics(self):
        """Test when there are no matching metric types."""
        slice_slos = [
            {
                "metric-type": "one-way-bandwidth",
                "bound": 1000
            }
        ]
        
        nrp_slos = {
            "slos": [
                {
                    "metric-type": "two-way-bandwidth",
                    "bound": 1500
                }
            ]
        }
        
        viable, score = slo_viability(slice_slos, nrp_slos)
        
        # Should still return True as no metrics failed
        assert viable is True
        assert score == 0
    
    def test_slo_viability_packet_loss_maximum_type(self):
        """Test packet loss as maximum constraint type."""
        slice_slos = [
            {
                "metric-type": "one-way-packet-loss",
                "bound": 0.01  # 1% maximum acceptable
            }
        ]
        
        nrp_slos = {
            "slos": [
                {
                    "metric-type": "one-way-packet-loss",
                    "bound": 0.005  # 0.5% NRP loss
                }
            ]
        }
        
        viable, score = slo_viability(slice_slos, nrp_slos)
        
        assert viable is True
        assert score > 0


class TestMapper:
    """Tests for mapper function."""
    
    def test_mapper_with_nrp_disabled_and_planner_disabled(self, app_context, sample_ietf_intent):
        """Test mapper when both NRP and Planner are disabled."""
        app_context.config = {
            "NRP_ENABLED": False,
            "PLANNER_ENABLED": False
        }
        
        result = mapper(sample_ietf_intent)
        
        assert result is None
    
    @patch('src.mapper.main.Planner')
    def test_mapper_with_planner_enabled(self, mock_planner_class, app_context, sample_ietf_intent):
        """Test mapper when Planner is enabled."""
        app_context.config = {
            "NRP_ENABLED": False,
            "PLANNER_ENABLED": True,
            "PLANNER_TYPE":"ENERGY"
        }
        
        mock_planner_instance = MagicMock()
        mock_planner_instance.planner.return_value = {"path": "node1->node2->node3"}
        mock_planner_class.return_value = mock_planner_instance
        
        result = mapper(sample_ietf_intent)
        
        assert result == {"path": "node1->node2->node3"}
        mock_planner_instance.planner.assert_called_once_with(sample_ietf_intent, "ENERGY")
    
    @patch('src.mapper.main.realizer')
    def test_mapper_with_nrp_enabled_finds_best_nrp(self, mock_realizer, app_context, sample_ietf_intent, sample_nrp_view):
        """Test mapper with NRP enabled finds the best NRP."""
        app_context.config = {
            "NRP_ENABLED": True,
            "PLANNER_ENABLED": False,
        }
        
        mock_realizer.return_value = sample_nrp_view
        
        result = mapper(sample_ietf_intent)
        
        # Verify realizer was called to READ NRP view
        assert mock_realizer.call_args_list[0] == call(None, True, "READ")
        assert result is None
    
    @patch('src.mapper.main.realizer')
    def test_mapper_with_nrp_enabled_no_viable_candidates(self, mock_realizer, app_context, sample_ietf_intent):
        """Test mapper when no viable NRPs are found."""
        app_context.config = {
            "NRP_ENABLED": True,
            "PLANNER_ENABLED": False
        }
        
        # All NRPs are unavailable
        nrp_view = [
            {
                "id": "nrp-1",
                "available": False,
                "slices": [],
                "slos": [
                    {
                        "metric-type": "one-way-bandwidth",
                        "bound": 500
                    }
                ]
            }
        ]
        
        mock_realizer.return_value = nrp_view
        
        result = mapper(sample_ietf_intent)
        
        assert result is None
    
    @patch('src.mapper.main.realizer')
    def test_mapper_with_nrp_enabled_creates_new_nrp(self, mock_realizer, app_context, sample_ietf_intent):
        """Test mapper creates new NRP when no suitable candidate exists."""
        app_context.config = {
            "NRP_ENABLED": True,
            "PLANNER_ENABLED": False
        }
        
        # No viable NRPs
        nrp_view = []
        
        mock_realizer.side_effect = [nrp_view, None]  # First call returns empty, second for CREATE
        
        result = mapper(sample_ietf_intent)
        
        # Verify CREATE was called
        create_call = [c for c in mock_realizer.call_args_list if len(c[0]) > 2 and c[0][2] == "CREATE"]
        assert len(create_call) > 0
    
    @patch('src.mapper.main.realizer')
    def test_mapper_with_nrp_and_planner_both_enabled(self, mock_realizer, app_context, sample_ietf_intent, sample_nrp_view):
        """Test mapper when both NRP and Planner are enabled."""
        app_context.config = {
            "NRP_ENABLED": True,
            "PLANNER_ENABLED": True,
            "PLANNER_TYPE":"ENERGY"
        }
        
        mock_realizer.return_value = sample_nrp_view
        
        with patch('src.mapper.main.Planner') as mock_planner_class:
            mock_planner_instance = MagicMock()
            mock_planner_instance.planner.return_value = {"path": "optimized_path"}
            mock_planner_class.return_value = mock_planner_instance
            
            result = mapper(sample_ietf_intent)
            
            # Planner should be called and return the result
            assert result == {"path": "optimized_path"}
    
    @patch('src.mapper.main.realizer')
    def test_mapper_updates_best_nrp_with_slice(self, mock_realizer, app_context, sample_ietf_intent, sample_nrp_view):
        """Test mapper updates best NRP with new slice."""
        app_context.config = {
            "NRP_ENABLED": True,
            "PLANNER_ENABLED": False
        }
        
        mock_realizer.return_value = sample_nrp_view
        
        result = mapper(sample_ietf_intent)
        
        # Verify UPDATE was called
        update_calls = [c for c in mock_realizer.call_args_list if len(c[0]) > 2 and c[0][2] == "UPDATE"]
        assert len(update_calls) > 0
    
    @patch('src.mapper.main.realizer')
    def test_mapper_extracts_slos_correctly(self, mock_realizer, app_context, sample_ietf_intent):
        """Test that mapper correctly extracts SLOs from intent."""
        app_context.config = {
            "NRP_ENABLED": True,
            "PLANNER_ENABLED": False
        }
        
        mock_realizer.return_value = []
        
        mapper(sample_ietf_intent)
        
        # Verify the function processed the intent
        assert mock_realizer.called
    
    @patch('src.mapper.main.logging')
    def test_mapper_logs_debug_info(self, mock_logging, app_context, sample_ietf_intent, sample_nrp_view):
        """Test mapper logs debug information."""
        app_context.config = {
            "NRP_ENABLED": True,
            "PLANNER_ENABLED": False
        }
        
        with patch('src.mapper.main.realizer') as mock_realizer:
            mock_realizer.return_value = sample_nrp_view
            
            mapper(sample_ietf_intent)
            
            # Verify debug logging was called
            assert mock_logging.debug.called


class TestMapperIntegration:
    """Integration tests for mapper functionality."""
    
    def test_mapper_complete_nrp_workflow(self, app_context, sample_ietf_intent, sample_nrp_view):
        """Test complete NRP mapping workflow."""
        app_context.config = {
            "NRP_ENABLED": True,
            "PLANNER_ENABLED": False
        }
        
        with patch('src.mapper.main.realizer') as mock_realizer:
            mock_realizer.return_value = sample_nrp_view
            
            result = mapper(sample_ietf_intent)
            
            # Verify the workflow sequence
            assert mock_realizer.call_count >= 1
            first_call = mock_realizer.call_args_list[0]
            assert first_call[0][1] is True  # need_nrp parameter
            assert first_call[0][2] == "READ"  # READ operation
    
    def test_mapper_complete_planner_workflow(self, app_context, sample_ietf_intent):
        """Test complete Planner workflow."""
        app_context.config = {
            "NRP_ENABLED": False,
            "PLANNER_ENABLED": True,
            "PLANNER_TYPE":"ENERGY"
        }
        
        expected_path = {
            "path": "node1->node2->node3",
            "cost": 10,
            "latency": 5
        }
        
        with patch('src.mapper.main.Planner') as mock_planner_class:
            mock_planner_instance = MagicMock()
            mock_planner_instance.planner.return_value = expected_path
            mock_planner_class.return_value = mock_planner_instance
            
            result = mapper(sample_ietf_intent)
            
            assert result == expected_path
            mock_planner_instance.planner.assert_called_once()
    
    def test_mapper_with_invalid_nrp_response(self, app_context, sample_ietf_intent):
        """Test mapper behavior with invalid NRP response."""
        app_context.config = {
            "NRP_ENABLED": True,
            "PLANNER_ENABLED": False
        }
        
        # Invalid NRP without expected fields
        invalid_nrp = {
            "id": "nrp-invalid"
            # Missing 'available' and 'slos' fields
        }
        
        with patch('src.mapper.main.realizer') as mock_realizer:
            mock_realizer.return_value = [invalid_nrp]
            
            # Should handle gracefully
            try:
                result = mapper(sample_ietf_intent)
            except (KeyError, TypeError):
                # Expected to fail gracefully
                pass
    
    def test_mapper_with_missing_slos_in_intent(self, app_context):
        """Test mapper behavior when intent has no SLOs."""
        app_context.config = {
            "NRP_ENABLED": True,
            "PLANNER_ENABLED": False
        }
        
        invalid_intent = {
            "ietf-network-slice-service:network-slice-services": {
                "slice-service": [{
                    "id": "slice-1"
                }],
                "slo-sle-templates": {
                    "slo-sle-template": [{
                        "id": "profile1",
                        "slo-policy": {
                            # No metric-bound key
                        }
                    }]
                }
            }
        }
        
        try:
            mapper(invalid_intent)
        except (KeyError, TypeError):
            # Expected behavior
            pass


class TestSloViabilityEdgeCases:
    """Edge case tests for slo_viability function."""
    
    def test_slo_viability_with_zero_bound(self):
        """Test handling of zero bounds in SLO."""
        slice_slos = [
            {
                "metric-type": "one-way-bandwidth",
                "bound": 0
            }
        ]
        
        nrp_slos = {
            "slos": [
                {
                    "metric-type": "one-way-bandwidth",
                    "bound": 100
                }
            ]
        }
        
        # Should handle zero division gracefully or fail as expected
        try:
            viable, score = slo_viability(slice_slos, nrp_slos)
        except (ZeroDivisionError, ValueError):
            pass
    
    def test_slo_viability_with_very_large_bounds(self):
        """Test handling of very large SLO bounds."""
        slice_slos = [
            {
                "metric-type": "one-way-bandwidth",
                "bound": 1e10
            }
        ]
        
        nrp_slos = {
            "slos": [
                {
                    "metric-type": "one-way-bandwidth",
                    "bound": 2e10
                }
            ]
        }
        
        viable, score = slo_viability(slice_slos, nrp_slos)
        
        assert viable is True
        assert isinstance(score, (int, float))
    
    def test_slo_viability_all_delay_types(self):
        """Test handling of all delay metric t ypes."""
        delay_types = [
            "one-way-delay-maximum",
            "two-way-delay-maximum",
            "one-way-delay-percentile",
            "two-way-delay-percentile",
            "one-way-delay-variation-maximum",
            "two-way-delay-variation-maximum"
        ]
        
        for delay_type in delay_types:
            slice_slos = [{"metric-type": delay_type, "bound": 10}]
            nrp_slos = {"slos": [{"metric-type": delay_type, "bound": 8}]}
            
            viable, score = slo_viability(slice_slos, nrp_slos)
            
            assert viable is True
            assert score >= 0