import json
import pytest
import os
from unittest.mock import patch, Mock, MagicMock
from pathlib import Path
from dotenv import load_dotenv
import sqlite3
import time
from flask import Flask
from src.main import NSController
from src.api.main import Api


# Load environment variables
load_dotenv()

@pytest.fixture(scope="session")
def flask_app():
    """Crea una app Flask mínima para los tests."""
    app = Flask(__name__)
    app.config.update({
        "TESTING": True,
        "SERVER_NAME": "localhost",
        'NRP_ENABLED': os.getenv('NRP_ENABLED', 'False').lower() == 'true',
        'PLANNER_ENABLED': os.getenv('PLANNER_ENABLED', 'False').lower() == 'true',
        'PCE_EXTERNAL': os.getenv('PCE_EXTERNAL', 'False').lower() == 'true',
        'DUMMY_MODE': os.getenv('DUMMY_MODE', 'True').lower() == 'true',
        'DUMP_TEMPLATES': os.getenv('DUMP_TEMPLATES', 'False').lower() == 'true',
        'TFS_L2VPN_SUPPORT': os.getenv('TFS_L2VPN_SUPPORT', 'False').lower() == 'true',
        'WEBUI_DEPLOY': os.getenv('WEBUI_DEPLOY', 'True').lower() == 'true',
        'UPLOAD_TYPE': os.getenv('UPLOAD_TYPE', 'WEBUI'),
        'PLANNER_TYPE': os.getenv('PLANNER_TYPE', 'ENERGY'),
        'HRAT_IP' : os.getenv('HRAT_IP', '10.0.0.1'),
        'OPTICAL_PLANNER_IP' : os.getenv('OPTICAL_PLANNER_IP', '10.0.0.1')
    })
    return app


@pytest.fixture(autouse=True)
def push_flask_context(flask_app):
    """Empuja automáticamente un contexto Flask para cada test."""
    with flask_app.app_context():
        yield

@pytest.fixture
def temp_db(tmp_path):
    """Fixture to create and cleanup test database using SQLite instead of JSON."""
    test_db_name = str(tmp_path / "test_slice.db")
    
    # Create database with proper schema
    conn = sqlite3.connect(test_db_name)
    cursor = conn.cursor()
    cursor.execute("""
        CREATE TABLE IF NOT EXISTS slice (
            slice_id TEXT PRIMARY KEY,
            intent TEXT NOT NULL,
            controller TEXT NOT NULL
        )
    """)
    conn.commit()
    conn.close()
    
    yield test_db_name
    
    # Cleanup - properly close connections and remove file
    try:
        time.sleep(0.1)
        if os.path.exists(test_db_name):
            os.remove(test_db_name)
    except Exception:
        time.sleep(0.5)
        try:
            if os.path.exists(test_db_name):
                os.remove(test_db_name)
        except:
            pass


@pytest.fixture
def env_variables():
    """Fixture to load and provide environment variables."""
    env_vars = {
        'NRP_ENABLED': os.getenv('NRP_ENABLED', 'False').lower() == 'true',
        'PLANNER_ENABLED': os.getenv('PLANNER_ENABLED', 'False').lower() == 'true',
        'PCE_EXTERNAL': os.getenv('PCE_EXTERNAL', 'False').lower() == 'true',
        'DUMMY_MODE': os.getenv('DUMMY_MODE', 'True').lower() == 'true',
        'DUMP_TEMPLATES': os.getenv('DUMP_TEMPLATES', 'False').lower() == 'true',
        'TFS_L2VPN_SUPPORT': os.getenv('TFS_L2VPN_SUPPORT', 'False').lower() == 'true',
        'WEBUI_DEPLOY': os.getenv('WEBUI_DEPLOY', 'True').lower() == 'true',
        'UPLOAD_TYPE': os.getenv('UPLOAD_TYPE', 'WEBUI'),
        'PLANNER_TYPE': os.getenv('PLANNER_TYPE', 'standard'),
    }
    return env_vars


@pytest.fixture
def controller_with_mocked_db(temp_db):
    """Crea un NSController con base de datos mockeada."""
    with patch('src.database.db.DB_NAME', temp_db):
        yield NSController(controller_type="TFS")


@pytest.fixture
def ietf_intent():
    """Intent válido en formato IETF."""
    return {
        "ietf-network-slice-service:network-slice-services": {
            "slo-sle-templates": {
                "slo-sle-template": [
                    {
                        "id": "qos1",
                        "slo-policy": {
                            "metric-bound": [
                                {
                                    "metric-type": "one-way-bandwidth",
                                    "metric-unit": "kbps",
                                    "bound": 1000
                                }
                            ]
                        }
                    }
                ]
            },
            "slice-service": [
                {
                    "id": "slice-test-1",
                    "sdps": {
                        "sdp": [
                            {
                                "sdp-ip-address": "10.0.0.1",
                                "node-id": "node1",
                                "service-match-criteria": {
                                    "match-criterion": [
                                        {
                                            "match-type": "vlan",
                                            "value": "100"
                                        }
                                    ]
                                },
                                "attachment-circuits": {
                                    "attachment-circuit": [
                                        {
                                            "sdp-peering": {
                                                "peer-sap-id": "R1"
                                            }
                                        }
                                    ]
                                },
                            },
                            {
                                "sdp-ip-address": "10.0.0.2",
                                "node-id": "node2",
                                "service-match-criteria": {
                                    "match-criterion": [
                                        {
                                            "match-type": "vlan",
                                            "value": "100"
                                        }
                                    ]
                                },
                                "attachment-circuits": {
                                    "attachment-circuit": [
                                        {
                                            "sdp-peering": {
                                                "peer-sap-id": "R2"
                                            }
                                        }
                                    ]
                                },
                            },
                        ]
                    },
                    "service-tags": {"tag-type": {"value": "L3VPN"}},
                }
            ],
        }
    }


class TestBasicApiOperations:
    """Tests for basic API operations."""
    
    def test_get_flows_empty(self, controller_with_mocked_db):
        """Debe devolver error cuando no hay slices."""
        result, code = Api(controller_with_mocked_db).get_flows()
        assert code == 404
        assert result["success"] is False
        assert result["data"] is None
    
    def test_add_flow_success(self, controller_with_mocked_db, ietf_intent):
        """Debe poder añadir un flow exitosamente."""
        with patch('src.database.db.save_data') as mock_save:
            result, code = Api(controller_with_mocked_db).add_flow(ietf_intent)
            assert code == 201
            assert result["success"] is True
            assert "slices" in result["data"]
    
    def test_add_and_get_flow(self, controller_with_mocked_db, ietf_intent):
        """Debe poder añadir un flow y luego recuperarlo."""
        with patch('src.database.db.save_data') as mock_save, \
             patch('src.database.db.get_all_data') as mock_get_all:
            
            Api(controller_with_mocked_db).add_flow(ietf_intent)
            
            mock_get_all.return_value = [
                {
                    "slice_id": "slice-test-1",
                    "intent": ietf_intent,
                    "controller": "TFS"
                }
            ]
            
            flows, code = Api(controller_with_mocked_db).get_flows()
            assert code == 200
            assert any(s["slice_id"] == "slice-test-1" for s in flows)
    
    def test_modify_flow_success(self, controller_with_mocked_db, ietf_intent):
        """Debe poder modificar un flow existente."""
        with patch('src.database.db.update_data') as mock_update:
            Api(controller_with_mocked_db).add_flow(ietf_intent)
            new_intent = ietf_intent.copy()
            new_intent["ietf-network-slice-service:network-slice-services"]["slo-sle-templates"]["slo-sle-template"][0]["id"] = "qos2"
            
            result, code = Api(controller_with_mocked_db).modify_flow("slice-test-1", new_intent)
            print(result)
            assert code == 200
            assert result["success"] is True
    
    def test_delete_specific_flow_success(self, controller_with_mocked_db, ietf_intent):
        """Debe borrar un flow concreto."""
        with patch('src.database.db.delete_data') as mock_delete:
            Api(controller_with_mocked_db).add_flow(ietf_intent)
            result, code = Api(controller_with_mocked_db).delete_flows("slice-test-1")
            assert code == 204
            assert result == {}
    
    def test_delete_all_flows_success(self, controller_with_mocked_db):
        """Debe borrar todos los flows."""
        with patch('src.database.db.delete_all_data') as mock_delete_all:
            result, code = Api(controller_with_mocked_db).delete_flows()
            assert code == 204
            assert result == {}
    
    def test_get_specific_flow(self, controller_with_mocked_db, ietf_intent):
        """Debe poder recuperar un flow específico."""
        with patch('src.database.db.get_data') as mock_get:
            Api(controller_with_mocked_db).add_flow(ietf_intent)
            mock_get.return_value = {
                "slice_id": "slice-test-1",
                "intent": ietf_intent,
                "controller": "TFS"
            }
            
            result, code = Api(controller_with_mocked_db).get_flows("slice-test-1")
            assert code == 200
            assert result["slice_id"] == "slice-test-1"


class TestErrorHandling:
    """Tests for error handling."""
    
    def test_add_flow_with_empty_intent(self, controller_with_mocked_db):
        """Debe fallar si se pasa un intent vacío."""
        result, code = Api(controller_with_mocked_db).add_flow({})
        assert code in (400, 404, 500)
        assert result["success"] is False
    
    def test_add_flow_with_none(self, controller_with_mocked_db):
        """Debe fallar si se pasa None como intent."""
        result, code = Api(controller_with_mocked_db).add_flow(None)
        assert code in (400, 500)
        assert result["success"] is False
    
    def test_get_nonexistent_slice(self, controller_with_mocked_db):
        """Debe devolver 404 si se pide un slice inexistente."""
        with patch('src.database.db.get_data') as mock_get:
            mock_get.side_effect = ValueError("No slice found")
            
            result, code = Api(controller_with_mocked_db).get_flows("slice-does-not-exist")
            assert code == 404
            assert result["success"] is False
    
    def test_modify_nonexistent_flow(self, controller_with_mocked_db, ietf_intent):
        """Debe fallar si se intenta modificar un flow inexistente."""
        with patch('src.database.db.update_data') as mock_update:
            mock_update.side_effect = ValueError("No slice found")
            
            result, code = Api(controller_with_mocked_db).modify_flow("nonexistent", ietf_intent)
            assert code == 404
            assert result["success"] is False
    
    def test_delete_nonexistent_flow(self, controller_with_mocked_db):
        """Debe fallar si se intenta eliminar un flow inexistente."""
        with patch('src.database.db.delete_data') as mock_delete:
            mock_delete.side_effect = ValueError("No slice found")
            
            result, code = Api(controller_with_mocked_db).delete_flows("nonexistent")
            assert code == 404
            assert result["success"] is False


