# Copyright 2022-2025 ETSI SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)

# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at

#     http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# This file includes original contributions from Telefonica Innovación Digital S.L.

import logging, requests, json
from src.config.constants import NBI_L2_PATH, NBI_L3_PATH

class tfs_connector(): 
    """
    Helper class to interact with TeraFlowSDN Northbound Interface (NBI) and WebUI.
    """  
    def webui_post(self, tfs_ip, service):
        """
        Post service descriptor to TFS WebUI.
        
        Args:
            tfs_ip (str): IP address of the TFS instance
            service (dict): Service descriptor to be posted
        
        Returns:
            requests.Response: Response object from the POST request
        """
        user="admin"
        password="admin"
        token=""
        session = requests.Session()
        session.auth = (user, password)
        url=f'http://{tfs_ip}/webui'
        response=session.get(url=url)
        for item in response.iter_lines():
            if("csrf_token" in str(item)):
                string=str(item).split('<input id="csrf_token" name="csrf_token" type="hidden" value=')[1]
                token=string.split(">")[0].strip('"')
        logging.debug("csrf token %s",token)

        files = {'descriptors': ("data.json", json.dumps(service).encode("utf-8"), "application/json")}
        token={'csrf_token':token}
        response = session.post(url,files=files,data=token,timeout=60)
        logging.debug("Http response: %s",response.text)
        return response

    def nbi_post(self, tfs_ip, service, path):
        """
        Post service descriptor to TFS NBI.
        Args:
            tfs_ip (str): IP address of the TFS instance
            service (dict): Service descriptor to be posted
            path (str): NBI endpoint path
        Returns:
            requests.Response: Response object from the POST request
        """
        token=""
        user="admin"
        password="admin"
        token=""
        session = requests.Session()
        session.auth = (user, password)
        url = f'http://{tfs_ip}/{path}'
        headers = {'Content-Type': 'application/json'}
        data = json.dumps(service)
        logging.debug("Posting to TFS NBI: %s",data)
        token={'csrf_token':token}
        response = session.post(url,headers=headers,data=data,timeout=60)
        response.raise_for_status()
        logging.debug("Http response: %s",response.text)
        return response
    
    def nbi_delete(self, tfs_ip: str, service_type: str , service_id: str) -> requests.Response:
        """
        Delete service from TFS NBI.
        Args:
            tfs_ip (str): IP address of the TFS instance
            service_type (str): Type of the service ('L2' or 'L3')
            service_id (str): Unique identifier of the service to delete
        Returns:
            requests.Response: Response object from the DELETE request
        """
        user="admin"
        password="admin"
        url = f'http://{user}:{password}@{tfs_ip}'
        if service_type == 'L2':
            url = url + f'/{NBI_L2_PATH}/vpn-service={service_id}'
        elif service_type == 'L3':
            url = url + f'/{NBI_L3_PATH}/vpn-service={service_id}'
        else:
            raise ValueError("Invalid service type. Use 'L2' or 'L3'.")
        response = requests.delete(url, timeout=60)
        response.raise_for_status()
        logging.debug('Service deleted successfully')
        logging.debug("Http response: %s",response.text)
        return response