# Copyright 2022-2025 ETSI SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)

# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at

#     http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# This file is an original contribution from Telefonica Innovación Digital S.L.

import logging
from netmiko import ConnectHandler

class cisco_connector():
    """Class to interact with Cisco devices via SSH using Netmiko."""
    def __init__(self, address, configs=None):
        self.address=address
        self.configs=configs

    def execute_commands(self, commands):
        """
        Execute a list of commands on the Cisco device.
        Args:
            commands (list): List of commands to execute on the device.
        """
        try:
            # Device configuration
            device = {
                'device_type': 'cisco_xr',  # This depends on the Cisco device type
                'host': self.address,
                'username': 'cisco',
                'password': 'cisco12345',
            }

            # SSH connection
            connection = ConnectHandler(**device)

            # Send commands
            output = connection.send_config_set(commands)
            logging.debug(output)

            # Close connection
            connection.disconnect()

        except Exception as e:
            logging.error(f"Failed to execute commands on {self.address}: {str(e)}")

    def create_command_template(self, config):
        """
        Create command template for configuring a Cisco device.

        Args:
            config (dict): Configuration parameters for the device.
        
        Returns:
            list: List of commands to configure the device.
        """
        commands = [
            "l2vpn",
            f"pw-class l2vpn_vpws_profile_example_{config['number']}",
            "encapsulation mpls"
        ]
        
        commands.extend([
            "transport-mode vlan passthrough",
            "control-word"
        ])
        
        commands.extend([
            f"preferred-path interface tunnel-te {config['number']}",
            "exit",
            "exit"
        ])
        
        commands.extend([
            "xconnect group l2vpn_vpws_group_example",
            f"p2p {config['ni_name']}",
            f"interface {config['interface']}.{config['vlan']}",
            f"neighbor ipv4 {config['remote_router']} pw-id {config['vlan']}",
            "no pw-class l2vpn_vpws_profile_example",
            f"pw-class l2vpn_vpws_profile_example_{config['number']}"
        ])
    

        return commands
    
    def full_create_command_template(self):
        """
        Create full command template for configuring a Cisco device based on the provided configurations.
        
        Returns:
            list: List of commands to configure the device.
        """
        commands =[]
        for config in self.configs:
            commands_temp = self.create_command_template(config)
            commands.extend(commands_temp)
        commands.append("commit")
        commands.append("end")
        return commands

    def create_command_template_delete(self):
        """
        Create command template for deleting L2VPN configuration on a Cisco device.
        Returns:
            list: List of commands to delete the L2VPN configuration.
        """
        commands = [
            "no l2vpn",
        ]
    
        commands.append("commit")
        commands.append("end")

        return commands