# Copyright 2022-2025 ETSI SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)

# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at

#     http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# This file is an original contribution from Telefonica Innovación Digital S.L.

import logging
from flask import current_app
from .tfs.tfs_connect import tfs_connect
from .ixia.ixia_connect import ixia_connect
from .e2e.e2e_connect import e2e_connect

def send_controller(controller_type, requests):
    """
    Route provisioning requests to the appropriate network controller.
    
    This function acts as a dispatcher that sends configuration requests to
    different SDN controller types based on the specified controller type.
    
    Args:
        controller_type (str): Type of controller to send requests to:
            - "TFS": TeraFlow SDN controller
            - "IXIA": Ixia network emulation controller
            - "E2E": TeraFlow End-to-End controller
        requests (dict or list): Configuration requests to be sent to the controller
    
    Returns:
        bool or dict: Response from the controller indicating success/failure
                     of the provisioning operation. Returns True in DUMMY_MODE.
                     
    Notes:
        - If DUMMY_MODE is enabled in config, returns True without sending requests
        - Uses IP addresses from Flask application configuration:
          * TFS_IP for TeraFlow
          * IXIA_IP for Ixia
          * TFS_E2E for End-to-End
        - Logs the controller type that received the request
        
    Raises:
        Exception: May be raised by individual connect functions on communication errors
    """
    if current_app.config["DUMMY_MODE"]:
        return True
        
    if controller_type == "TFS":
        response = tfs_connect(requests, current_app.config["TFS_IP"])
        logging.info("Request sent to Teraflow")
    elif controller_type == "IXIA":
        response = ixia_connect(requests, current_app.config["IXIA_IP"])
        logging.info("Requests sent to Ixia")
    elif controller_type == "E2E":
        response = e2e_connect(requests, current_app.config["TFS_E2E"])
        logging.info("Requests sent to Teraflow E2E")
        
    return response