# Copyright 2022-2025 ETSI SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)

# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at

#     http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# This file is an original contribution from Telefonica Innovación Digital S.L.

import logging
from .ixia.main import ixia
from .tfs.main  import tfs
from .e2e.main  import e2e

def select_way(controller=None, way=None, ietf_intent=None, response=None, rules = None):
    """
    Determine the method of slice realization.

    Args:
        controller (str): The controller to use for slice realization. Defaults to None.
            Supported values:
            - "IXIA": IXIA NEII for network testing
            - "TFS": TeraFlow Service for network slice management
            - "E2E": End-to-End controller for e2e slice management
        way (str): The type of technology to use. Defaults to None.
        ietf_intent (dict): IETF-formatted network slice intent. Defaults to None.
        response (dict): Response built for user feedback. Defaults to None.
        rules (list, optional): Specific rules for slice realization. Defaults to None.

    Returns:
        dict: A realization request for the specified network slice type.

    """
    realizing_request = None
    if controller == "TFS":
        realizing_request = tfs(ietf_intent, way, response)
    elif controller == "IXIA":
        realizing_request = ixia(ietf_intent)
    elif controller == "E2E":
        realizing_request = e2e(ietf_intent, way, response, rules)
    else:
        logging.warning(f"Unsupported controller: {controller}. Defaulting to TFS realization.")
        realizing_request = tfs(ietf_intent, way, response)
    return realizing_request