# Copyright 2022-2025 ETSI SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)

# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at

#     http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# This file includes original contributions from Telefonica Innovación Digital S.L.

import logging
from .select_way import select_way
from .nrp_handler import nrp_handler
from src.utils.safe_get import safe_get

def realizer(ietf_intent, need_nrp=False, order=None, nrp=None, controller_type=None, response=None, rules = None):
    """
    Manage the slice creation workflow.

    This method handles two primary scenarios:
    1. Interact with network controllers for NRP (Network Resource Partition) operations when need_nrp is True
    2. Slice service selection when need_nrp is False

    Args:
        ietf_intent (dict): IETF-formatted network slice intent.
        need_nrp (bool, optional): Flag to indicate if NRP operations are needed. Defaults to False.
        order (str, optional): Type of NRP operation (READ, UPDATE, CREATE). Defaults to None.
        nrp (dict, optional): Specific Network Resource Partition to operate on. Defaults to None.
        controller_type (str, optional): Type of controller (TFS, IXIA, E2E). Defaults to None.
        response (dict, optional): Response built for user feedback. Defaults to None.
        rules (dict, optional): Specific rules for slice realization. Defaults to None.
    
    Returns:
        dict: A realization request for the specified network slice type.
    """
    if need_nrp:
        # Perform NRP-related operations
        nrp_view = nrp_handler(order, nrp)
        return nrp_view
    else:
        # Select slice service method
        if controller_type == "E2E":
            if isinstance(rules, list) and len(rules) > 0: rules = rules[0]
            actions = rules.get("actions", []) if (rules and not type(rules)== str) else []

            has_transceiver  = any(a.get("type", "").startswith("XR_AGENT_ACTIVATE_TRANSCEIVER") for a in actions)
            has_optical      = any(a.get("type", "").startswith("PROVISION_MEDIA_CHANNEL") for a in actions)
            has_l3           = any(a.get("type", "").startswith("CONFIG_VPNL3") for a in actions)
            has_l2           = any(a.get("type", "").startswith("CONFIG_VPNL2") for a in actions)

            del_transceiver  = any(a.get("type", "").startswith("DEACTIVATE_XR_AGENT_TRANSCEIVER") for a in actions)
            del_optical      = any(a.get("type", "").startswith("DEPROVISION_OPTICAL_RESOURCE") for a in actions)
            del_l3           = any(a.get("type", "").startswith("REMOVE_VPNL3") for a in actions)
            del_l2           = any(a.get("type", "").startswith("REMOVE_VPNL2") for a in actions)

            if   has_transceiver:         selected_way = "L3oWDM"
            elif has_optical and has_l3:  selected_way = "L3oWDM"
            elif has_optical and has_l2:  selected_way = "L2oWDM"
            elif has_optical:             selected_way = "OPTIC"
            elif has_l3:                  selected_way = "L3VPN"
            elif has_l2:                  selected_way = "L2VPN"

            elif del_transceiver:         selected_way = "DEL_L3oWDM"
            elif del_optical and del_l3:  selected_way = "DEL_L3oWDM"
            elif del_optical and del_l2:  selected_way = "DEL_L2oWDM"
            elif del_optical:             selected_way = "DEL_OPTIC"
            elif del_l3:                  selected_way = "DEL_L3VPN"
            elif del_l2:                  selected_way = "DEL_L2VPN"
            else:
                logging.warning("Cannot determine the realization way from rules. Skipping request.")
                return None
            way = selected_way
        else:
            way = safe_get(ietf_intent, ['ietf-network-slice-service:network-slice-services', 'slice-service', 0, 'service-tags', 'tag-type', 0, 'tag-type-value', 0])
        logging.info(f"Selected way: {way}")
        request = select_way(controller=controller_type, way=way, ietf_intent=ietf_intent, response=response, rules = rules)
        return request
