# Copyright 2022-2025 ETSI SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)

# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at

#     http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# This file is an original contribution from Telefonica Innovación Digital S.L.

import logging

def ixia(ietf_intent):
    """
    Prepare an Ixia service request based on the IETF intent.

    This method configures an Ixia service request by:
    1. Defining endpoint routers
    2. Loading a service template
    3. Generating a unique service UUID
    4. Configuring service endpoints
    5. Adding QoS constraints

    Args:
        ietf_intent (dict): IETF-formatted network slice intent.

    Returns:
        dict: An Ixia service request for configuration.
    """
    metric_bounds = ietf_intent.get("ietf-network-slice-service:network-slice-services", {}) \
        .get("slo-sle-templates", {}) \
        .get("slo-sle-template", [{}])[0] \
        .get("slo-policy", {}) \
        .get("metric-bound", [])

    # Inicializar valores
    bandwidth = None
    latency = None
    tolerance = None

    # Asignar valores según el tipo de métrica
    for metric in metric_bounds:
        metric_type = metric.get("metric-type")
        bound = metric.get("bound")

        if metric_type == "one-way-bandwidth":
            bandwidth = bound
        elif metric_type == "one-way-delay-maximum":
            latency = bound
        elif metric_type == "one-way-delay-variation-maximum": 
            tolerance = bound

    # Construcción del diccionario intent
    intent = {
        "src_node_ip": ietf_intent.get("ietf-network-slice-service:network-slice-services", {})
            .get("slice-service", [{}])[0]
            .get("sdps", {}).get("sdp", [{}])[0]
            .get("attachment-circuits", {}).get("attachment-circuit", [{}])[0]
            .get("sdp-peering", {}).get("peer-sap-id"),

        "dst_node_ip": ietf_intent.get("ietf-network-slice-service:network-slice-services", {})
            .get("slice-service", [{}])[0]
            .get("sdps", {}).get("sdp", [{}, {}])[1]
            .get("attachment-circuits", {}).get("attachment-circuit", [{}])[0]
            .get("sdp-peering", {}).get("peer-sap-id"),

        "vlan_id": ietf_intent.get("ietf-network-slice-service:network-slice-services", {})
            .get("slice-service", [{}])[0]
            .get("sdps", {}).get("sdp", [{}])[0]
            .get("service-match-criteria", {}).get("match-criterion", [{}])[0]
            .get("value"),

        "bandwidth": bandwidth,
        "latency": latency,
        "tolerance": tolerance,

        "latency_version": ietf_intent.get("ietf-network-slice-service:network-slice-services", {})
            .get("slo-sle-templates", {}).get("slo-sle-template", [{}])[0]
            .get("description"),

        "reliability": ietf_intent.get("ietf-network-slice-service:network-slice-services", {})
            .get("slo-sle-templates", {}).get("slo-sle-template", [{}])[0]
            .get("sle-policy", {}).get("reliability"),
    }

    logging.info(f"IXIA Intent realized\n")
    return intent