# Copyright 2022-2025 ETSI SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)

# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at

#     http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# This file is an original contribution from Telefonica Innovación Digital S.L.

import requests
class automatizacion:
    def obtener_informacion_ip(ip):
        '''
        Gets the information IP of the NE2
        Args:
            ip: IP.
        Returns:
            A dictionary with the IP infotmationx.
        '''
        url= "http://"+ip+"/api/actions/ipInfo"
        body= {"ip": ip}
        response = requests.get(url, json=body, auth=('admin', 'admin'))
        if response.status_code==200:
            return response.json()
        else:
            print(f"error al obtener la informacion de la IP; {response.status_code}")
            return None
        
    def obtener_informacion_hardware(ip):
        """
        Obtiene información de una dirección IP.

        Args:
            ip: La dirección IP del NE2.

        Returns:
            Un diccionario con la información de IP.
        """
        url = "http://"+ip+"/api/actions/hwInfo"
        body = {"ip": ip}
        response = requests.get(url, json=body, auth=('admin', 'admin'))
        if response.status_code == 200:
            return response.json()
        else:
            print(f"Error al obtener la información de la IP: {response.status_code}")
            return None
        
    def obtener_informacion_puerto(ip,puerto):
        """
        Obtiene información de una dirección IP.

        Args:
            ip: La dirección IP del NE2.

        Returns:
            Un diccionario con la información de IP.
        """
        url = "http://"+ip+"/api/hw/Port/"+puerto
        body = {"ip": ip}
        response = requests.get(url, json=body, auth=('admin', 'admin'))
        if response.status_code == 200:
            return response.json()
        else:
            print(f"Error al obtener la información de la IP: {response.status_code}")
            return None
        
    def añadir_configuracion_puerto_delay(delay, latency_type, max_latency):
        """
        Añade una configuración de puerto según su delay a un NE2.

        Args:
            delay (int): Cantidad de delay en la simulación.

        Returns:
            La respuesta de la API (texto).
        """
        configuracion=None
        print(f'\n\nTipo de latencia: {latency_type}\n latencia: {delay}\n')
        if latency_type=='1' or latency_type==None:
            configuracion={'ethernetDelay': {'delay': delay, 'delayMax': 15.0, 'isUncorrelated': False, 'maxNegDelta': 0.1, 'pdvMode': 'NONE', 'delayMin': 5.0, 'units': 'MS', 'maxPosDelta': 0.1, 'enabled': True, 'spread': 1.0}}
        if latency_type=='2' or latency_type=='gauss':
            delay_f=float(delay)
            ancho=float(max_latency)
            max_delay=delay_f+ancho
            min_delay=delay_f-ancho
            configuracion={'ethernetDelay': {'delay': delay, 'delayMax': max_delay, 'isUncorrelated': False,'maxNegDelta': ancho/3, 'pdvMode': 'GAUSSIAN', 'delayMin':min_delay, 'units': 'MS', 'maxPosDelta':ancho/3, 'enabled': True, 'spread': 1.58}}
        if latency_type =='3' or latency_type=='internet':
            ancho=float(max_latency)
            max_delay = float(delay)+0.9*float(ancho)
            min_delay = float(delay)-0.1*float(ancho)
            configuracion={'ethernetDelay': {'delay': delay, 'delayMax': max_delay, 'isUncorrelated': False, 'maxNegDelta': 0.4, 'pdvMode': 'INTERNET', 'delayMin': min_delay, 'units': 'MS', 'maxPosDelta': 0.5, 'enabled': True, 'spread': 100.0}}
        print(f"\n\nConf Delay: {configuracion}\n")    
        return configuracion
        
    def añadir_configuracion_puerto_ipv4(source_hx,destination_hx,protocolo):
        """
        Añade una configuración de puerto según las IPv4s de origen y destino de la comunicación.

        Args:
            source_hx (int): Dirección IPv4 de origen en hexadecimal.
            destination_hx (int): Dirección IPv4 de destino en hexadecimal.
            protocolo (string): Numero asociado al protocolo elegido.

        Returns:
            La respuesta de la API (texto).
        """
        reglas = [{"field": "Common::IPv4::Version", "value": "4", "mask": "f"}]
        if source_hx:
            reglas.append({"field": "Common::IPv4::Source Address", "value": source_hx, "mask": "ffffffff"})
        if destination_hx:
            reglas.append({"field": "Common::IPv4::Destination Address", "value": destination_hx, "mask": "ffffffff"})
        if protocolo:
            reglas.append({"field": "Common::IPv4::Protocol", "value": protocolo, "mask": "ff"})
        configuracion = {"profiles": [{"dramAllocation": {"mode": "AUTO","fixedSize": 1700352},"rules": reglas,"tag": "test_api"}],"defaultProfile": {"dramAllocation": {"mode": "AUTO","fixedSize": 1700352},"tag": "defaultProfile"}}
        return configuracion

    def añadir_configuracion_puerto_ipv6(source,destination):
        """
        Añade una configuración de puerto según las IPv6s de origen y destino de la comunicación.

        Args:
            source (int): Dirección IPv6 de origen en hexadecimal.
            destination (int): Dirección IPv6 de destino en hexadecimal.

        Returns:
            La respuesta de la API (texto).
        """
        reglas=[{'field': 'Common::IPv6::Version', 'bitRange': 'L3@0[7]+3', 'value': '6', 'mask': 'f'}]
        if source:
            reglas.append({'field': 'Common::IPv6::Source Address', 'bitRange': 'L3@8[7]+127', 'value': source, 'mask': 'ffffffffffffffffffffffffffffffff'})
        if destination:
            reglas.append({'field': 'Common::IPv6::Destination Address', 'bitRange': 'L3@24[7]+127', 'value': destination, 'mask': 'ffffffffffffffffffffffffffffffff'})
        configuracion = {"profiles": [{"dramAllocation": {"mode": "AUTO","fixedSize": 1700352},"rules": reglas,"tag": "test_api"}],"defaultProfile": {"dramAllocation": {"mode": "AUTO","fixedSize": 1700352},"tag": "defaultProfile"}}
        return configuracion

    def añadir_configuracion_VLAN(vlan):
        """
        Añade una configuración de puerto según su VLAN.

        Args:
            vlan (int): Número de identificación de la VLAN.

        Returns:
            La respuesta de la API (texto).
        """
        vlan=hex(vlan)[2:]
        configuracion={'profiles': [{'dramAllocation': {'mode': 'AUTO', 'fixedSize': 1700352}, 'rules': [{'field': 'Common::Second Tag::VLAN ID', 'bitRange': 'L2@18[3]+11', 'value': vlan, 'mask': 'fff'}]}]}
        return configuracion

    def añadir_configuración_packetDrop(drops,total):
        """
        Añade una configuración de puerto según su configuracion de packet Drop un NE2.

        Args:
            drops (int): cantidad de paquetes dropeados
            total (int): cantidad total de paquetes

        Returns:
            La respuesta de la API (texto).
        """
        configuracion={'packetDrop': {'rdmSel': {'dist': 'PERIODIC', 'burstlen': drops, 'interval': total, 'stddev': 10.0}, 'enabled': True}}
        return configuracion

    def añadir_configuracion_policer(bitrate):
        """
        Añade una configuración de puerto según su TX bandwidth a un NE2.

        Args:
            bitrate (int): bandwith en Tx

        Returns:
            La respuesta de la API (texto).
        """
        configuracion={'policer':{'excessBitRate': bitrate, 'excessBurstTolerance': 64000, 'commitedBurstTolerance': 64000, 'commitedBitRate': bitrate, 'enabled': True, 'enableRateCoupling': False}}
        return configuracion
    
    def añadir_configuracion_shaper(bitrate):
        """
        Añade una configuración de puerto según su RX bandwidth a un NE2.

        Args:
            bitrate (int): bandwith en Rx

        Returns:
            La respuesta de la API (texto).
        """
        configuracion={'shaper': {'burstTolerance': 64000, 'bitRate': bitrate, 'enabled': True}}
        return configuracion
    
    def añadir_configuracion_reorder(reorder):
        """
        Adds reorder configuration.

        Args:
            reorder (int): reorder

        Returns:
            response of the API (texto).
        """
        reorder=100-reorder
        print(f"\nReorder:{reorder}\n")
        configuracion={"reorder": {"rdmSel": {"dist": "PERIODIC", "burstlen": reorder, "interval": 100, "stddev": 10.0 }, "reorderByMin": 1, "reorderByMax": 5, "enabled": True}} #Los demas valores los he dejado como defecto por no poder configurarlos
        return configuracion

    def envio_peticion(ip,puerto, configuracion):
        """
        Envía una configuración de puerto a un NE2.

        Args:
            ip (int): IP del NE2
            puerto: número del puerto a configurar
            configuración: la información que se quiere configurar en el puerto

        Returns:
            La respuesta de la API (texto).
        """

import requests
class automatizacion:
    def obtener_informacion_ip(ip):
        '''
        obtiene informacion ip del NE2
        Args:
            ip: la direccion IP del NE2.

        Returns:
            un diccionario con la informacion de IP.
        '''
        url= "http://"+ip+"/api/actions/ipInfo"
        body= {"ip": ip}
        response = requests.get(url, json=body, auth=('admin', 'admin'))
        if response.status_code==200:
            return response.json()
        else:
            print(f"error al obtener la informacion de la IP; {response.status_code}")
            return None
        
    def obtener_informacion_hardware(ip):
        """
        Obtiene información de una dirección IP.

        Args:
            ip: La dirección IP del NE2.

        Returns:
            Un diccionario con la información de IP.
        """
        url = "http://"+ip+"/api/actions/hwInfo"
        body = {"ip": ip}
        response = requests.get(url, json=body, auth=('admin', 'admin'))
        if response.status_code == 200:
            return response.json()
        else:
            print(f"Error al obtener la información de la IP: {response.status_code}")
            return None
        
    def obtener_informacion_puerto(ip,puerto):
        """
        Obtiene información de una dirección IP.

        Args:
            ip: La dirección IP del NE2.

        Returns:
            Un diccionario con la información de IP.
        """
        url = "http://"+ip+"/api/hw/Port/"+puerto
        body = {"ip": ip}
        response = requests.get(url, json=body, auth=('admin', 'admin'))
        if response.status_code == 200:
            return response.json()
        else:
            print(f"Error al obtener la información de la IP: {response.status_code}")
            return None
        
    def añadir_configuracion_puerto_delay(delay, latency_type, max_latency):
        """
        Añade una configuración de puerto según su delay a un NE2.

        Args:
            delay (int): Cantidad de delay en la simulación.

        Returns:
            La respuesta de la API (texto).
        """
        configuracion=None
        print(f'\n\nTipo de latencia: {latency_type}\n latencia: {delay}\n')
        if latency_type=='1' or latency_type==None:
            configuracion={'ethernetDelay': {'delay': delay, 'delayMax': 15.0, 'isUncorrelated': False, 'maxNegDelta': 0.1, 'pdvMode': 'NONE', 'delayMin': 5.0, 'units': 'MS', 'maxPosDelta': 0.1, 'enabled': True, 'spread': 1.0}}
        if latency_type=='2' or latency_type=='gauss':
            delay_f=float(delay)
            ancho=float(max_latency)
            max_delay=delay_f+ancho
            min_delay=delay_f-ancho
            configuracion={'ethernetDelay': {'delay': delay, 'delayMax': max_delay, 'isUncorrelated': False,'maxNegDelta': ancho/3, 'pdvMode': 'GAUSSIAN', 'delayMin':min_delay, 'units': 'MS', 'maxPosDelta':ancho/3, 'enabled': True, 'spread': 1.58}}
        if latency_type =='3' or latency_type=='internet':
            ancho=float(max_latency)
            max_delay = float(delay)+0.9*float(ancho)
            min_delay = float(delay)-0.1*float(ancho)
            configuracion={'ethernetDelay': {'delay': delay, 'delayMax': max_delay, 'isUncorrelated': False, 'maxNegDelta': 0.4, 'pdvMode': 'INTERNET', 'delayMin': min_delay, 'units': 'MS', 'maxPosDelta': 0.5, 'enabled': True, 'spread': 100.0}}
        print(f"\n\nConf Delay: {configuracion}\n")    
        return configuracion
        
    def añadir_configuracion_puerto_ipv4(source_hx,destination_hx,protocolo):
        """
        Añade una configuración de puerto según las IPv4s de origen y destino de la comunicación.

        Args:
            source_hx (int): Dirección IPv4 de origen en hexadecimal.
            destination_hx (int): Dirección IPv4 de destino en hexadecimal.
            protocolo (string): Numero asociado al protocolo elegido.

        Returns:
            La respuesta de la API (texto).
        """
        reglas = [{"field": "Common::IPv4::Version", "value": "4", "mask": "f"}]
        if source_hx:
            reglas.append({"field": "Common::IPv4::Source Address", "value": source_hx, "mask": "ffffffff"})
        if destination_hx:
            reglas.append({"field": "Common::IPv4::Destination Address", "value": destination_hx, "mask": "ffffffff"})
        if protocolo:
            reglas.append({"field": "Common::IPv4::Protocol", "value": protocolo, "mask": "ff"})
        configuracion = {"profiles": [{"dramAllocation": {"mode": "AUTO","fixedSize": 1700352},"rules": reglas,"tag": "test_api"}],"defaultProfile": {"dramAllocation": {"mode": "AUTO","fixedSize": 1700352},"tag": "defaultProfile"}}
        return configuracion

    def añadir_configuracion_puerto_ipv6(source,destination):
        """
        Añade una configuración de puerto según las IPv6s de origen y destino de la comunicación.

        Args:
            source (int): Dirección IPv6 de origen en hexadecimal.
            destination (int): Dirección IPv6 de destino en hexadecimal.

        Returns:
            La respuesta de la API (texto).
        """
        reglas=[{'field': 'Common::IPv6::Version', 'bitRange': 'L3@0[7]+3', 'value': '6', 'mask': 'f'}]
        if source:
            reglas.append({'field': 'Common::IPv6::Source Address', 'bitRange': 'L3@8[7]+127', 'value': source, 'mask': 'ffffffffffffffffffffffffffffffff'})
        if destination:
            reglas.append({'field': 'Common::IPv6::Destination Address', 'bitRange': 'L3@24[7]+127', 'value': destination, 'mask': 'ffffffffffffffffffffffffffffffff'})
        configuracion = {"profiles": [{"dramAllocation": {"mode": "AUTO","fixedSize": 1700352},"rules": reglas,"tag": "test_api"}],"defaultProfile": {"dramAllocation": {"mode": "AUTO","fixedSize": 1700352},"tag": "defaultProfile"}}
        return configuracion

    def añadir_configuracion_VLAN(vlan):
        """
        Añade una configuración de puerto según su VLAN.

        Args:
            vlan (int): Número de identificación de la VLAN.

        Returns:
            La respuesta de la API (texto).
        """
        vlan=hex(vlan)[2:]
        configuracion={'profiles': [{'dramAllocation': {'mode': 'AUTO', 'fixedSize': 1700352}, 'rules': [{'field': 'Common::Second Tag::VLAN ID', 'bitRange': 'L2@18[3]+11', 'value': vlan, 'mask': 'fff'}]}]}
        return configuracion

    def añadir_configuración_packetDrop(drops,total,version,desv):
        """
        Añade una configuración de puerto según su configuracion de packet Drop un NE2.

        Args:
            drops (int): cantidad de paquetes dropeados
            total (int): cantidad total de paquetes
            version (string): version of the probability

        Returns:
            La respuesta de la API (texto).
        """
        if version != "GAUSSIAN" and version !="POISSON":
            configuracion={'packetDrop': {'rdmSel': {'dist': version, 'burstlen': drops, 'interval': total, 'stddev': 10.0}, 'enabled': True}}
        else:
            configuracion={'packetDrop': {'rdmSel': {'dist': version, 'burstlen': drops, 'interval': total, 'stddev': desv}, 'enabled': True}}
        return configuracion

    def añadir_configuracion_policer(bitrate):
        """
        Añade una configuración de puerto según su TX bandwidth a un NE2.

        Args:
            bitrate (int): bandwith en Tx

        Returns:
            La respuesta de la API (texto).
        """
        configuracion={'policer':{'excessBitRate': bitrate, 'excessBurstTolerance': 64000, 'commitedBurstTolerance': 64000, 'commitedBitRate': bitrate, 'enabled': True, 'enableRateCoupling': False}}
        return configuracion
    
    def añadir_configuracion_shaper(bitrate):
        """
        Añade una configuración de puerto según su RX bandwidth a un NE2.

        Args:
            bitrate (int): bandwith en Rx

        Returns:
            La respuesta de la API (texto).
        """
        configuracion={'shaper': {'burstTolerance': 64000, 'bitRate': bitrate, 'enabled': True}}
        return configuracion
    
    def añadir_configuracion_reorder(packages,reorder,npackagesreorder, maxreord, version, stev):
        """
        Adds reorder configuration.

        Args:
            packages: number of packages total
            reorder: number of packages to reorder
            npackagesreorder: number of packages of reorder
            maxreord: total number of packages of reorder
            version: version of reorder
            stev: desviation

        Returns:
            response of the API (text).
        """
        if packages is not None and npackagesreorder is not None and maxreord is not None: 
            reorderByMin = min(npackagesreorder, maxreord)
            reorderByMax = max(npackagesreorder, maxreord)
            if version != 'GAUSSIAN':
                configuracion={"reorder": {"rdmSel": {"dist": version, "burstlen": reorder, "interval": packages, "stddev": 10.0 }, "reorderByMin": reorderByMin, "reorderByMax": reorderByMax, "enabled": True}} 
            else:
                configuracion={"reorder": {"rdmSel": {"dist": "GAUSSIAN", "burstlen": reorder, "interval": packages, "stddev": stev }, "reorderByMin": reorderByMin, "reorderByMax": reorderByMax, "enabled": True}} 
            if int(reorder) <= 0:
                configuracion["filterWarning"] = "El valor de reorder es inválido, está fuera del rango permitido."
        else:
            reorder=10000-reorder
            configuracion={"reorder": {"rdmSel": {"dist": "PERIODIC", "burstlen": int(reorder/100), "interval": 100, "stddev": 10.0 }, "reorderByMin": 1, "reorderByMax": 5, "enabled": True}}
        return configuracion

    def envio_peticion(ip,puerto, configuracion):
        """
        Envía una configuración de puerto a un NE2.

        Args:
            ip (int): IP del NE2
            puerto: número del puerto a configurar
            configuración: la información que se quiere configurar en el puerto

        Returns:
            La respuesta de la API (texto).
        """
        print(f'\nCONFIGURACION\n{configuracion}')
        url = f"http://{ip}/api/hw/Port/{puerto}"
        response = requests.put(url, json=configuracion, auth=('admin', 'admin'))

        if response.status_code == 200:
            print(f'\n{configuracion}')
            return response.text
        else:
            try:
                error_info = response.json()
            except ValueError:
                error_info = response.text

            print(f"\n\nError al añadir configuración de puerto: {response.status_code}")
            print(f"Mensaje de error: {error_info}")
            print(f"Configuración enviada: {configuracion}\n\n")
            return None