# Copyright 2022-2025 ETSI SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)

# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at

#     http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# This file includes original contributions from Telefonica Innovación Digital S.L.

import logging, os
from src.config.constants import TEMPLATES_PATH
from src.utils.load_template import load_template

def l3ipowdm_slice(rules):
    """
    Prepare a Optical service request for an optical slice.

    This method prepares a TeraFlow service request for an optical slice by:
    1. Defining endpoint routers
    2. Loading a service template
    3. Generating a unique service UUID
    4. Configuring service endpoints
    5. Adding QoS constraints

    Args:
        ietf_intent (dict): IETF-formatted network slice intent.
        rules (dict, optional): Configuration rules for the optical slice.

    Returns:
        dict: A TeraFlow service request for optical slice configuration.
    """
    transceiver_params = []
    bandwidth = 0

    logging.debug(f"Preparing L3oWDM slice with rules: {rules}")
    tfs_requests = []
    for rule in rules["actions"]:
        logging.debug(f"Processing rule: {rule['type']}")
        if rule["type"] == "CREATE_OPTICAL_SLICE":
            tfs_request = load_template(os.path.join(TEMPLATES_PATH, "Optical_slice.json"))
            request = optical_slice_template(tfs_request, rules)
            logging.debug(f"Sending Optical Slice to Optical Controller {request}")
            tfs_requests.append(request)

        elif rule["type"] == "PROVISION_MEDIA_CHANNEL_OLS_PATH":

            origin_router_id         = rule["content"]["src-sip-uuid"]
            destination_router_id    = rule["content"]["dest-sip-uuid"]
            direction                = rule["content"]["direction"]
            bandwidth                = rule["content"]["bandwidth-ghz"]
            service_uuid             = rule["content"]["ols-path-uuid"]
            tenant_uuid              = rule["tenant-uuid"]
            layer_protocol_name      = rule["content"]["layer-protocol-name"]
            layer_protocol_qualifier = rule["content"]["layer-protocol-qualifier"]
            lower_frequency_mhz      = rule["content"]["lower-frequency-mhz"]
            upper_frequency_mhz      = rule["content"]["upper-frequency-mhz"]
            link_uuid_path           = rule["content"]["link-uuid-path"]
            granularity              = rule["content"]["adjustment-granularity"]
            grid                     = rule["content"]["grid-type"]

            tfs_request = load_template(os.path.join(TEMPLATES_PATH, "TAPI_service.json"))

            tfs_request["services"][0]["service_id"]["service_uuid"]["uuid"] = service_uuid
            config_rules = tfs_request["services"][0]["service_config"]["config_rules"][0]

            config_rules["tapi_lsp"]["rule_set"]["src"]  = origin_router_id
            config_rules["tapi_lsp"]["rule_set"]["dst"]  = destination_router_id
            config_rules["tapi_lsp"]["rule_set"]["uuid"] = service_uuid
            config_rules["tapi_lsp"]["rule_set"]["bw"]   = str(bandwidth)
            config_rules["tapi_lsp"]["rule_set"]["tenant_uuid"] = tenant_uuid
            config_rules["tapi_lsp"]["rule_set"]["direction"]   = direction
            config_rules["tapi_lsp"]["rule_set"]["layer_protocol_name"] = layer_protocol_name
            config_rules["tapi_lsp"]["rule_set"]["layer_protocol_qualifier"] = layer_protocol_qualifier
            config_rules["tapi_lsp"]["rule_set"]["lower_frequency_mhz"] = str(lower_frequency_mhz)
            config_rules["tapi_lsp"]["rule_set"]["upper_frequency_mhz"] = str(upper_frequency_mhz)
            config_rules["tapi_lsp"]["rule_set"]["link_uuid_path"] = link_uuid_path
            config_rules["tapi_lsp"]["rule_set"]["granularity"]    = granularity
            config_rules["tapi_lsp"]["rule_set"]["grid_type"]      = grid

            logging.debug(f"Sending Media Channel Service to Orchestrator: {tfs_request}")
            tfs_requests.append(tfs_request)

        elif rule["type"] == "ACTIVATE_TRANSCEIVER":
            params = {
                "router_id": rule["content"]["node-uuid"],
                "router_tp": rule["content"]["termination-point-uuid"],
                "frequency": rule["content"]["frequency-ghz"],
                "power":     rule["content"]["tx-power-dbm"]
            }
            transceiver_params.append(params)
        elif rule["type"] == "CONFIG_VPNL3":
            src_router_id  = rule["content"]["src-node-uuid"]

            if src_router_id == transceiver_params[0]["router_id"]:
                src_power = transceiver_params[0]["power"]
                src_frequency = transceiver_params[0]["frequency"]
                dst_power = transceiver_params[1]["power"]
                dst_frequency = transceiver_params[1]["frequency"]
            else:
                src_power = transceiver_params[1]["power"]
                src_frequency = transceiver_params[1]["frequency"]
                dst_power = transceiver_params[0]["power"]
                dst_frequency = transceiver_params[0]["frequency"]

            src_router_id  = rule["content"]["src-node-uuid"]
            src_ip_address = rule["content"]["src-ip-address"]
            src_ip_mask    = rule["content"]["src-ip-mask"]
            src_vlan_id    = rule["content"]["src-vlan-id"]

            dst_router_id  = rule["content"]["dest-node-uuid"]
            dst_ip_address = rule["content"]["dest-ip-address"]
            dst_ip_mask    = rule["content"]["dest-ip-mask"]
            dst_vlan_id    = rule["content"]["dest-vlan-id"]

            service_uuid = rule["content"]["tunnel-uuid"]

            tfs_request = load_template(os.path.join(TEMPLATES_PATH, "IPoWDM_orchestrator.json"))
            tfs_request["services"][0]["service_id"]["service_uuid"]["uuid"] = service_uuid
            config_rules = tfs_request["services"][0]["service_config"]["config_rules"][0]
            src = config_rules["ipowdm"]["rule_set"]["src"]
            src.append({
                'uuid': src_router_id,
                'ip_address': src_ip_address,
                'ip_mask': src_ip_mask,
                'vlan_id': src_vlan_id,
                'power': src_power,
                'frequency': src_frequency
            })

            dst = config_rules["ipowdm"]["rule_set"]["dst"]
            dst.append({
                'uuid': dst_router_id,
                'ip_address': dst_ip_address,
                'ip_mask': dst_ip_mask,
                'vlan_id': dst_vlan_id,
                'power': dst_power,
                'frequency': dst_frequency
            })

            config_rules["ipowdm"]["rule_set"]["bw"]        = bandwidth
            config_rules["ipowdm"]["rule_set"]["uuid"]      = service_uuid

            logging.debug(f"Sending IPoWDM Service to Orchestrator: {tfs_request}")
            tfs_requests.append(tfs_request)

        else:
            logging.debug("Unsupported rule type for optical slice: %s", rule["type"])
    return tfs_requests

def optical_slice_template(template, rule):
    """
    Complete the optical slice template with the data provided.
    Args:
        template (dict): optical slice template.
        data (dict): Data to complete the template.
    Returns:
        dict: Template completed.
    """

    for action in rule.get('actions', []):
        content = action.get('content', {})
        nodes = content.get('node', [])
        for node in nodes:
            for onp in node.get('owned-node-edge-point', []):
                if 'media-channel-node-edge-point-spec' in onp:
                    onp['tapi-photonic-media:media-channel-node-edge-point-spec'] = onp.pop('media-channel-node-edge-point-spec')

    for i, sip in enumerate(template['tapi-common:context']['service-interface-point']):
        if i < len(rule['actions'][0]['content']['service-interface-point']):
            sip['uuid'] = rule['actions'][0]['content']['service-interface-point'][i]['uuid']

    nodes_template = template['tapi-common:context']['tapi-topology:topology-context']['topology'][0]['node']
    nodes_data = rule['actions'][0]['content']['node']
    for new_node in nodes_data:
        nodes_template.append(new_node)

    links_template = template['tapi-common:context']['tapi-topology:topology-context']['topology'][0]['link']
    links_rule = rule['actions'][0]['content']['link']
    for link_t in links_rule:
        links_template.append(link_t)

    template['tapi-common:context']['uuid'] = rule['actions'][0]['content']['tenant-uuid']
    template['tapi-common:context']['name'][0]['value'] = rule['network-slice-uuid']

    return template
