# Copyright 2022-2025 ETSI SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)

# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at

#     http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# This file is an original contribution from Telefonica Innovación Digital S.L.

import logging
from src.planner.planner import Planner
from .slo_viability import slo_viability
from src.realizer.main import realizer
from flask import current_app

def mapper(ietf_intent):
    """
    Map an IETF network slice intent to the most suitable Network Resource Partition (NRP).

    This method:
    1. If NRP is enabled, retrieves the current NRP view
    2. Extracts Service Level Objectives (SLOs) from the intent
    3. Finds NRPs that can meet the SLO requirements
    4. Selects the best NRP based on viability and availability
    5. Attaches the slice to the selected NRP or creates a new one
    6. If planner is enabled, computes the optimal path for the slice

    Args:
        ietf_intent (dict): IETF-formatted network slice intent.

    Returns:
        dict or None: Optimal path if planner is enabled; otherwise, None.
    """
    if current_app.config["NRP_ENABLED"]:
        # Retrieve NRP view
        nrp_view = realizer(None, True, "READ")

        # Extract Service Level Objectives (SLOs) from the intent
        slos = ietf_intent["ietf-network-slice-service:network-slice-services"]["slo-sle-templates"]["slo-sle-template"][0]["slo-policy"]["metric-bound"]
        if slos:
            # Find candidate NRPs that can meet the SLO requirements
            candidates = [
                (nrp, slo_viability(slos, nrp)[1])
                for nrp in nrp_view
                if slo_viability(slos, nrp)[0] and nrp["available"]
            ]
            logging.debug(f"Candidates: {candidates}")

            # Select the best NRP based on candidates
            best_nrp = max(candidates, key=lambda x: x[1])[0] if candidates else None
            logging.debug(f"Best NRP: {best_nrp}")

            if best_nrp:
                best_nrp["slices"].append(ietf_intent["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["id"])
                # Update NRP view
                realizer(ietf_intent, True, "UPDATE")
                # TODO Here we should put how the slice is attached to an already created nrp
            else:
                # Request the controller to create a new NRP that meets the SLOs
                answer = realizer(ietf_intent, True, "CREATE", best_nrp)
                if not answer:
                    logging.error("Slice rejected due to lack of NRPs")
                    return None
                # TODO Here we should put how the slice is attached to the new nrp

    if current_app.config["PLANNER_ENABLED"]:
        optimal_path = Planner().planner(ietf_intent, current_app.config["PLANNER_TYPE"])
        logging.debug(f"Optimal path: {optimal_path}")
        return optimal_path
    return None