# Copyright 2022-2025 ETSI SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)

# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at

#     http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import sqlite3, json, logging

# Database file
DB_NAME = "slice.db"

# Initialize database and create table
def init_db():
    """
    Initialize the SQLite database and create the slice table if not exists.
    """
    conn = sqlite3.connect(DB_NAME)
    cursor = conn.cursor()
    cursor.execute("""
        CREATE TABLE IF NOT EXISTS slice (
            slice_id TEXT PRIMARY KEY,
            intent TEXT NOT NULL,
            controller TEXT NOT NULL
        )
    """)
    conn.commit()
    conn.close()

# Save data to the database
def save_data(slice_id: str, intent_dict: dict, controller: str):
    """
    Save a new slice entry to the database.

    Args:
        slice_id (str): Unique identifier for the slice
        intent_dict (dict): Intent data
        controller (str): Controller type
    
    Raises:
        ValueError: If a slice with the given slice_id already exists
    """
    intent_str = json.dumps(intent_dict)
    conn = sqlite3.connect(DB_NAME)
    cursor = conn.cursor()
    try:
        cursor.execute("INSERT INTO slice (slice_id, intent, controller) VALUES (?, ?, ?)", (slice_id, intent_str, controller))
        conn.commit()
    # Handle duplicate slice ID
    except sqlite3.IntegrityError:
        raise ValueError(f"Slice with id '{slice_id}' already exists.")
    finally:
        conn.close()

# Update data in the database
def update_data(slice_id: str, new_intent_dict: dict, controller: str):
    """
    Update an existing slice entry in the database.

    Args:
        slice_id (str): Unique identifier for the slice
        new_intent_dict (dict): New intent data
        controller (str): Controller type
    
    Raises:
        ValueError: If no slice is found with the given slice_id
    """
    intent_str = json.dumps(new_intent_dict)
    conn = sqlite3.connect(DB_NAME)
    cursor = conn.cursor()
    cursor.execute("UPDATE slice SET intent = ?, controller = ? WHERE slice_id = ?", (intent_str, controller, slice_id))
    if cursor.rowcount == 0:
        raise ValueError(f"No slice found with id '{slice_id}' to update.")
    else:
        logging.debug(f"Slice '{slice_id}' updated.")
    conn.commit()
    conn.close()

# Delete data from the database
def delete_data(slice_id: str):
    """
    Delete a slice entry from the database.

    Args:
        slice_id (str): Unique identifier for the slice to delete
    
    Raises:
        ValueError: If no slice is found with the given slice_id
    """
    conn = sqlite3.connect(DB_NAME)
    cursor = conn.cursor()
    cursor.execute("DELETE FROM slice WHERE slice_id = ?", (slice_id,))
    if cursor.rowcount == 0:
        raise ValueError(f"No slice found with id '{slice_id}' to delete.")
    else:
        logging.debug(f"Slice '{slice_id}' deleted.")
    conn.commit()
    conn.close()

# Get data from the database
def get_data(slice_id: str) -> dict[str, dict, str]:
    """
    Retrieve a specific slice entry from the database.

    Args:
        slice_id (str): Unique identifier for the slice to retrieve
    
    Returns:
        dict: Slice data including slice_id, intent (as dict), and controller
    
    Raises:
        ValueError: If no slice is found with the given slice_id
        Exception: For JSON decoding errors
    """
    conn = sqlite3.connect(DB_NAME)
    cursor = conn.cursor()
    cursor.execute("SELECT * FROM slice WHERE slice_id = ?", (slice_id,))
    row = cursor.fetchone()
    conn.close()

    if row:
        column_names = [description[0] for description in cursor.description]
        result = dict(zip(column_names, row))
        if isinstance(result.get("intent"), str):
            try:
                result["intent"] = json.loads(result["intent"])
            except json.JSONDecodeError:
                raise Exception("Warning: 'intent' is not a valid JSON string.")
        return result

    else:
        raise ValueError(f"No slice found with id '{slice_id}'.")

# Get all slices
def get_all_data() -> dict[str, dict, str]:
    """
    Retrieve all slice entries from the database.

    Returns:
        list: List of slice data dictionaries including slice_id, intent (as dict), and controller
    """
    conn = sqlite3.connect(DB_NAME)
    cursor = conn.cursor()
    cursor.execute("SELECT * FROM slice")
    rows = cursor.fetchall()
    conn.close()
    return [
        {
        "slice_id": row[0],
        "intent": json.loads(row[1]),
        "controller": row[2] 
        }
    for row in rows
    ]

def delete_all_data():
    """
    Delete all slice entries from the database.
    """
    conn = sqlite3.connect(DB_NAME)
    cursor = conn.cursor()
    cursor.execute("DELETE FROM slice")
    conn.commit()
    conn.close()
    logging.debug("All slice data deleted.")

# Example usage
if __name__ == "__main__":
    init_db()

    # Save a slice
    test_intent = {"bandwidth": "1Gbps", "latency": "10ms", "provider": "opensec"}
    save_data("slice-001", test_intent, "TFS")

    # Get the slice
    result = get_data("slice-001")
    if result:
        print(f"Retrieved intent for slice-001: {result}")

    # Update the slice
    updated_intent = {"bandwidth": "2Gbps", "latency": "5ms", "provider": "opensec"}
    update_data("slice-001", updated_intent, "TFS")

    # Delete the slice
    delete_data("slice-001")

    get_all_data()
    delete_all_data()
