# Copyright 2022-2025 ETSI SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)

# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at

#     http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# This file is an original contribution from Telefonica Innovación Digital S.L.

from src.utils.send_response import send_response
import logging
from flask import current_app
from src.database.db import get_data, delete_data, get_all_data, delete_all_data
from src.realizer.tfs.helpers.tfs_connector import tfs_connector
from src.utils.safe_get import safe_get

class Api:
    def __init__(self, slice_service):
        self.slice_service = slice_service
    
    def add_flow(self, intent):
        """
        Create a new transport network slice.

        Args:
            intent (dict): Network slice intent in 3GPP or IETF format

        Returns:
            Result of the Network Slice Controller (NSC) operation

        API Endpoint:
            POST /slice

        Raises:
            RuntimeError: If there is no content to process
            Exception: For unexpected errors
        """
        try:
            result = self.slice_service.nsc(intent)
            if not result:
                return send_response(False, code=404, message="No intents found")
            logging.info(f"Slice created successfully")
            return send_response(
                True,
                code=201,
                data=result 
            )
        except RuntimeError as e:
            # Handle case where there is no content to process
            return send_response(False, code=200, message=str(e))
        except Exception as e:
            # Handle unexpected errors
            return send_response(False, code=500, message=str(e))
    
    def get_flows(self,slice_id=None):
        """
        Retrieve transport network slice information.

        This method allows retrieving:
        - All transport network slices
        - A specific slice by its ID

        Args:
            slice_id (str, optional): Unique identifier of a specific slice. 
                                      Defaults to None.

        Returns:
            dict or list: 
            - If slice_id is provided: Returns the specific slice details
            - If slice_id is None: Returns a list of all slices
            - Returns an error response if no slices are found

        API Endpoint:
            GET /slice/{id}

        Raises:
            ValueError: If no transport network slices are found
            Exception: For unexpected errors
        """
        try:
            # Read slice database from JSON file
            content = get_all_data()
            # If specific slice ID is provided, find and return matching slice
            if slice_id:
                for slice in content:
                    if slice["slice_id"] == slice_id:
                        return slice, 200
                raise ValueError("Transport network slices not found")
            # If no slices exist, raise an error
            if len(content) == 0:
                raise ValueError("Transport network slices not found")
            
            # Return all slices if no specific ID is given
            return [slice for slice in content if slice.get("controller") == self.slice_service.controller_type], 200
        
        except ValueError as e:
            # Handle case where no slices are found
            return send_response(False, code=404, message=str(e))
        except Exception as e:
            # Handle unexpected errors
            return send_response(False, code=500, message=str(e))
    
    def modify_flow(self,slice_id, intent):
        """
        Modify an existing transport network slice.

        Args:
            slice_id (str): Unique identifier of the slice to modify
            intent (dict): New intent configuration for the slice

        Returns:
            Result of the Network Slice Controller (NSC) operation

        API Endpoint:
            PUT /slice/{id}
        Raises:
            Exception: For unexpected errors
        """
        try:
            result = self.slice_service.nsc(intent, slice_id)
            if not result:
                return send_response(False, code=404, message="Slice not found")
            logging.info(f"Slice {slice_id} modified successfully")
            return send_response(
                True,
                code=200,
                message="Slice modified successfully",
                data=result
            )
        except ValueError as e:
            # Handle case where no slices are found
            return send_response(False, code=404, message=str(e))
        except Exception as e:
            # Handle unexpected errors
            return send_response(False, code=500, message=str(e))
    
    def delete_flows(self, slice_id=None):
        """
        Delete transport network slice(s).

        This method supports:
        - Deleting a specific slice by ID
        - Deleting all slices
        - Optional cleanup of L2VPN configurations

        Args:
            slice_id (str, optional): Unique identifier of slice to delete. 
                                      Defaults to None.

        Returns:
            dict: {} indicating successful deletion or error details

        API Endpoint:
            DELETE /slice/{id}

        Raises:
            ValueError: If no slices are found to delete
            Exception: For unexpected errors

        Notes:
            - If controller_type is TFS, attempts to delete from Teraflow
            - If need_l2vpn_support is True, performs additional L2VPN cleanup
        """
        try:
            # Delete specific slice if slice_id is provided
            if slice_id:
                slice = get_data(slice_id)
                # Raise error if slice not found
                if not slice or slice.get("controller") != self.slice_service.controller_type:
                    raise ValueError("Transport network slice not found")
                # Delete in Teraflow
                if not current_app.config["DUMMY_MODE"]:
                    if self.slice_service.controller_type == "TFS":
                        slice_type = safe_get(slice, ['intent', 'ietf-network-slice-service:network-slice-services', 'slice-service', 0, 'service-tags', 'tag-type', 0, 'tag-type-value', 0])
                        if not slice_type:
                            slice_type = "L2"
                            logging.warning(f"Slice type not found in slice intent. Defaulting to L2")
                        tfs_connector().nbi_delete(current_app.config["TFS_IP"],slice_type, slice_id)
                # Update slice database
                delete_data(slice_id)
                logging.info(f"Slice {slice_id} removed successfully")
                return {}, 204
            
            # Delete all slices
            else:
                # Optional: Delete in Teraflow if configured
                if not current_app.config["DUMMY_MODE"]:
                    if self.slice_service.controller_type == "TFS":
                        content = get_all_data()
                        for slice in content:
                            if slice.get("controller") == self.slice_service.controller_type:
                                slice_type = safe_get(slice, ['intent', 'ietf-network-slice-service:network-slice-services', 'slice-service', 0, 'service-tags', 'tag-type', 0, 'tag-type-value', 0])
                                if not slice_type:
                                    slice_type = "L2"
                                    logging.warning(f"Slice type not found in slice intent. Defaulting to L2")
                                tfs_connector().nbi_delete(current_app.config["TFS_IP"],slice_type, slice.get("slice_id"))
                        if current_app.config["TFS_L2VPN_SUPPORT"]:
                            self.slice_service.tfs_l2vpn_delete()

                # Clear slice database
                delete_all_data()

                logging.info("All slices removed successfully")
                return {}, 204
        
        except ValueError as e:
            return send_response(False, code=404, message=str(e))
        except Exception as e:
            return send_response(False, code=500, message=str(e))