# Copyright 2022-2025 ETSI SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)

# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at

#     http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# This file is an original contribution from Telefonica Innovación Digital S.L.

import json, os
from src.config.constants import TEMPLATES_PATH
from flask import current_app

def dump_templates(nbi_file, ietf_file, realizer_file):
    """
    Dump multiple template files as JSON for debugging and analysis.
    
    This utility function saves network slice templates at different processing
    stages to disk for inspection, debugging, and documentation purposes.
    Only executes if DUMP_TEMPLATES configuration flag is enabled.

    Args:
        nbi_file (dict): Northbound Interface template - original user/API request
        ietf_file (dict): IETF-standardized network slice intent format
        realizer_file (dict): Controller-specific realization template
        
    Returns:
        None
        
    Notes:
        - Controlled by DUMP_TEMPLATES configuration flag
        - Files saved to TEMPLATES_PATH directory
        - Output files:
          * nbi_template.json - Original NBI request
          * ietf_template.json - Standardized IETF format
          * realizer_template.json - Controller-specific format
        - JSON formatted with 2-space indentation for readability
        - Silently returns if DUMP_TEMPLATES is False
        
    Raises:
        IOError: If unable to write to TEMPLATES_PATH directory
    """
    if not current_app.config["DUMP_TEMPLATES"]:
        return

    # Map template content to output filenames
    templates = {
        "nbi_template.json": nbi_file,
        "ietf_template.json": ietf_file,
        "realizer_template.json": realizer_file,
    }

    # Write each template to disk
    for filename, content in templates.items():
        path = os.path.join(TEMPLATES_PATH, filename)
        with open(path, "w") as f:
            json.dump(content, f, indent=2)