# Copyright 2022-2025 ETSI SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)

# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at

#     http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# This file is an original contribution from Telefonica Innovación Digital S.L.

from .helpers.tfs_connector import tfs_connector
from flask import current_app
from src.utils.send_response import send_response
from .service_types.tfs_l2vpn import tfs_l2vpn_support

def tfs_connect(requests, tfs_ip):
    """
    Connect to TeraflowSDN (TFS) controller and upload services.
    
    Args:
        requests (dict): Dictionary containing services to upload
        tfs_ip (str): IP address of the TFS controller
    
    Returns:
        response (requests.Response): Response from TFS controller
    """       
    if current_app.config["UPLOAD_TYPE"] == "WEBUI":
        response = tfs_connector().webui_post(tfs_ip, requests)
    elif current_app.config["UPLOAD_TYPE"] == "NBI":
        for intent in requests["services"]:
            # Send each separate NBI request
            path = intent.pop("path")
            response = tfs_connector().nbi_post(tfs_ip, intent, path)

            if not response.ok:
                return send_response(False, code=response.status_code, message=f"Teraflow upload failed. Response: {response.text}")
    
    # For deploying an L2VPN with path selection (not supported by Teraflow)
    if current_app.config["TFS_L2VPN_SUPPORT"]:
        tfs_l2vpn_support(requests["services"])
    
    return response