# Copyright 2022-2025 ETSI SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)

# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at

#     http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# This file includes original contributions from Telefonica Innovación Digital S.L.

import logging, os
from src.config.constants import TEMPLATES_PATH, NBI_L2_PATH
from src.utils.load_template import load_template
from src.utils.safe_get import safe_get
from ..helpers.cisco_connector import cisco_connector
from flask import current_app

def tfs_l2vpn(ietf_intent, response):
    """
    Translate slice intent into a TeraFlow service request.

    This method prepares a L2VPN service request by:
    1. Defining endpoint routers
    2. Loading a service template
    3. Generating a unique service UUID
    4. Configuring service endpoints
    5. Adding QoS constraints
    6. Preparing configuration rules for network interfaces

    Args:
        ietf_intent (dict): IETF-formatted network slice intent.
        response (dict): Response data containing slice information.

    Returns:
        dict: A TeraFlow service request for L2VPN configuration.

    """
    # Hardcoded router endpoints
    # TODO (should be dynamically determined)
    origin_router_id = safe_get(ietf_intent, ["ietf-network-slice-service:network-slice-services", "slice-service", 0, "sdps", "sdp", 0, "attachment-circuits", "attachment-circuit", 0, "sdp-peering", "peer-sap-id"])
    if not origin_router_id:
        logging.warning("Origin router ID not found in the intent. Skipping L2VPN realization.")
        return None
    origin_router_if = '0/0/0-GigabitEthernet0/0/0/0'
    destination_router_id = safe_get(ietf_intent, ["ietf-network-slice-service:network-slice-services", "slice-service", 0, "sdps", "sdp", 1, "attachment-circuits", "attachment-circuit", 0, "sdp-peering", "peer-sap-id"])
    if not destination_router_id:
        logging.warning("Destination router ID not found in the intent. Skipping L2VPN realization.")
        return None
    destination_router_if = '0/0/0-GigabitEthernet0/0/0/0'
    id = ietf_intent["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["id"]
    slice = next((d for d in response if d.get("id") == id), None)

    if current_app.config["UPLOAD_TYPE"] == "WEBUI":
        # Load L2VPN service template
        tfs_request = load_template(os.path.join(TEMPLATES_PATH, "L2-VPN_template_empty.json"))["services"][0]

        # Configure service UUID
        tfs_request["service_id"]["service_uuid"]["uuid"] = ietf_intent['ietf-network-slice-service:network-slice-services']['slice-service'][0]["id"]

        # Configure service endpoints
        for endpoint in tfs_request["service_endpoint_ids"]:
            endpoint["device_id"]["device_uuid"]["uuid"] = origin_router_id if endpoint is tfs_request["service_endpoint_ids"][0] else destination_router_id
            endpoint["endpoint_uuid"]["uuid"] = origin_router_if if endpoint is tfs_request["service_endpoint_ids"][0] else destination_router_if

        # Add service constraints
        for constraint in slice.get("requirements", []):
            tfs_request["service_constraints"].append({"custom": constraint})

        # Add configuration rules
        for i, config_rule in enumerate(tfs_request["service_config"]["config_rules"][1:], start=1):
            router_id = origin_router_id if i == 1 else destination_router_id
            router_if = origin_router_if if i == 1 else destination_router_if
            resource_value = config_rule["custom"]["resource_value"]

            sdp_index = i - 1
            vlan_value = ietf_intent["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["sdps"]["sdp"][sdp_index]["service-match-criteria"]["match-criterion"][0]["value"]
            if vlan_value:
                resource_value["vlan_id"] = int(vlan_value)
            resource_value["circuit_id"] = vlan_value
            resource_value["remote_router"] = destination_router_id if i == 1 else origin_router_id
            resource_value["ni_name"] = 'ELAN{:s}'.format(str(vlan_value))
            config_rule["custom"]["resource_key"] = f"/device[{router_id}]/endpoint[{router_if}]/settings"

    elif current_app.config["UPLOAD_TYPE"] == "NBI":
        #self.path = NBI_L2_PATH
        # Load IETF L2VPN service template
        tfs_request = load_template(os.path.join(TEMPLATES_PATH, "ietfL2VPN_template_empty.json"))
        
        # Add path to the request
        tfs_request["path"] = NBI_L2_PATH

        # Generate service UUID
        full_id = ietf_intent["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["id"]
        uuid_only = full_id.split("slice-service-")[-1]
        tfs_request["ietf-l2vpn-svc:vpn-service"][0]["vpn-id"] = uuid_only

        # Configure service endpoints
        sites = tfs_request["ietf-l2vpn-svc:vpn-service"][0]["site"]
        sdps = ietf_intent["ietf-network-slice-service:network-slice-services"]["slice-service"][0]["sdps"]["sdp"]

        for i, site in enumerate(sites):
            is_origin = (i == 0)
            router_id = origin_router_id if is_origin else destination_router_id
            sdp = sdps[0] if is_origin else sdps[1]
            site["site-id"] = router_id
            site["site-location"] = sdp["node-id"]
            site["site-network-access"]["interface"]["ip-address"] = sdp["sdp-ip-address"]

    logging.info(f"L2VPN Intent realized")
    return tfs_request

def tfs_l2vpn_support(requests):
    """
    Configuration support for L2VPN with path selection based on MPLS traffic-engineering tunnels

    Args:
        requests (list): A list of configuration parameters.

    """
    sources={
        "source": "10.60.125.44",
        "config":[]
    }
    destinations={
        "destination": "10.60.125.45",
        "config":[]
    }
    for request in requests:
        # Configure Source Endpoint
        temp_source = request["service_config"]["config_rules"][1]["custom"]["resource_value"]
        endpoints = request["service_endpoint_ids"]
        config = {
            "ni_name": temp_source["ni_name"],
            "remote_router": temp_source["remote_router"],
            "interface": endpoints[0]["endpoint_uuid"]["uuid"].replace("0/0/0-", ""),
            "vlan" : temp_source["vlan_id"],
            "number" : temp_source["vlan_id"] % 10 + 1
        }
        sources["config"].append(config)

        # Configure Destination Endpoint
        temp_destiny = request["service_config"]["config_rules"][2]["custom"]["resource_value"]
        config = {
            "ni_name": temp_destiny["ni_name"],
            "remote_router": temp_destiny["remote_router"],
            "interface": endpoints[1]["endpoint_uuid"]["uuid"].replace("0/0/3-", ""),
            "vlan" : temp_destiny["vlan_id"],
            "number" : temp_destiny["vlan_id"] % 10 + 1
        }
        destinations["config"].append(config)
        
    #cisco_source = cisco_connector(source_address, ni_name, remote_router, vlan, vlan % 10 + 1)
    cisco_source = cisco_connector(sources["source"], sources["config"])
    commands = cisco_source.full_create_command_template()
    cisco_source.execute_commands(commands)

    #cisco_destiny = cisco_connector(destination_address, ni_name, remote_router, vlan, vlan % 10 + 1)
    cisco_destiny = cisco_connector(destinations["destination"], destinations["config"])
    commands = cisco_destiny.full_create_command_template()
    cisco_destiny.execute_commands(commands)

def tfs_l2vpn_delete():
    """
    Delete L2VPN configurations from Cisco devices.

    This method removes L2VPN configurations from Cisco routers

    Notes:
        - Uses cisco_connector to generate and execute deletion commands
        - Clears Network Interface (NI) settings
    """
    # Delete Source Endpoint Configuration
    source_address = "10.60.125.44"
    cisco_source = cisco_connector(source_address)
    cisco_source.execute_commands(cisco_source.create_command_template_delete())

    # Delete Destination Endpoint Configuration
    destination_address = "10.60.125.45"
    cisco_destiny = cisco_connector(destination_address)
    cisco_destiny.execute_commands(cisco_destiny.create_command_template_delete())