# Copyright 2022-2025 ETSI SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)

# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at

#     http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# This file is an original contribution from Telefonica Innovación Digital S.L.

import logging
from src.planner.energy_planner.energy           import energy_planner
from src.planner.hrat_planner.hrat               import hrat_planner
from src.planner.tfs_optical_planner.tfs_optical import tfs_optical_planner
from flask import current_app


class Planner:
    """
    Planner class to compute optimal paths for network slices.
    Uses different strategies based on configuration.
    """
    """
    Planner class to compute the optimal path for a network slice based on energy consumption and topology.
    """

    def planner(self, intent, type):
        """
        Plan the optimal path for a network slice based on energy consumption and topology.

        Args:
            intent (dict): Network slice intent
            type (str): Planner type (ENERGY, HRAT, TFS_OPTICAL)

        Returns:
            dict or None: Planner result or None if type is invalid
        """
        # Log selected planner type
        logging.info(f"Planner type selected: {type}")
        # Use energy planner strategy
        if type   == "ENERGY"     : return energy_planner(intent)
        # Use HRAT planner with configured IP
        elif type == "HRAT"       : return hrat_planner(intent, current_app.config["HRAT_IP"])
        # Use TFS optical planner with configured IP
        elif type == "TFS_OPTICAL": return tfs_optical_planner(intent, current_app.config["OPTICAL_PLANNER_IP"], action = "create")
        # Return None if planner type is unsupported
        else : return None
