# Copyright 2022-2023 ETSI TeraFlowSDN - TFS OSG (https://tfs.etsi.org/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import os
os.environ['DEVICE_EMULATED_ONLY'] = 'YES'

# pylint: disable=wrong-import-position
import logging, pytest, time
from typing import Dict
from device.service.drivers.gnmi_openconfig.GnmiOpenConfigDriver import GnmiOpenConfigDriver
from .storage.Storage import Storage
from .tools.check_config import (
    check_config_endpoints, check_config_interfaces, check_config_network_instances
)
from .tools.check_updates import check_updates
from .tools.request_composers import (
    interface, network_instance, network_instance_interface, network_instance_static_route
)

logging.basicConfig(level=logging.DEBUG)
LOGGER = logging.getLogger(__name__)
LOGGER.setLevel(logging.DEBUG)


##### DRIVER FIXTURE ###################################################################################################

DRIVER_SETTING_ADDRESS  = '172.20.20.101'
DRIVER_SETTING_PORT     = 6030
DRIVER_SETTING_USERNAME = 'admin'
DRIVER_SETTING_PASSWORD = 'admin'
DRIVER_SETTING_USE_TLS  = False

@pytest.fixture(scope='session')
def driver() -> GnmiOpenConfigDriver:
    _driver = GnmiOpenConfigDriver(
        DRIVER_SETTING_ADDRESS, DRIVER_SETTING_PORT,
        username=DRIVER_SETTING_USERNAME,
        password=DRIVER_SETTING_PASSWORD,
        use_tls=DRIVER_SETTING_USE_TLS,
    )
    _driver.Connect()
    yield _driver
    time.sleep(1)
    _driver.Disconnect()


##### STORAGE FIXTURE ##################################################################################################

@pytest.fixture(scope='session')
def storage() -> Dict:
    yield Storage()


##### NETWORK INSTANCE DETAILS #########################################################################################

NETWORK_INSTANCES = [
    {
        'name': 'test-l3-svc',
        'type': 'L3VRF',
        'interfaces': [
            {'name': 'Ethernet1',  'subif_index': 0, 'ipv4_addr': '192.168.1.1',  'ipv4_prefix': 24, 'enabled': True},
            {'name': 'Ethernet10', 'subif_index': 0, 'ipv4_addr': '192.168.10.1', 'ipv4_prefix': 24, 'enabled': True},
        ],
        'static_routes': [
            {'prefix': '172.0.0.0/24', 'gateway': '172.16.0.2', 'metric': 1},
            {'prefix': '172.2.0.0/24', 'gateway': '172.16.0.3', 'metric': 1},
        ]
    },
    {
        'name': 'test-l2-svc',
        'type': 'L2VSI',
        'interfaces': [
            {'name': 'Ethernet2',  'subif_index': 0, 'ipv4_addr': '192.168.1.1',  'ipv4_prefix': 24, 'enabled': True},
            {'name': 'Ethernet4', 'subif_index': 0, 'ipv4_addr': '192.168.10.1', 'ipv4_prefix': 24, 'enabled': True},
        ],
        'static_routes': [
            {'prefix': '172.0.0.0/24', 'gateway': '172.16.0.2', 'metric': 1},
            {'prefix': '172.2.0.0/24', 'gateway': '172.16.0.3', 'metric': 1},
        ]
    }
]


##### TEST METHODS #####################################################################################################

def test_get_endpoints(
    driver : GnmiOpenConfigDriver,  # pylint: disable=redefined-outer-name
    storage : Storage,              # pylint: disable=redefined-outer-name
) -> None:
    results_getconfig = check_config_endpoints(driver, storage)
    storage.endpoints.populate(results_getconfig)
    check_config_endpoints(driver, storage)


def test_get_interfaces(
    driver : GnmiOpenConfigDriver,  # pylint: disable=redefined-outer-name
    storage : Storage,              # pylint: disable=redefined-outer-name
) -> None:
    results_getconfig = check_config_interfaces(driver, storage)
    storage.interfaces.populate(results_getconfig)
    check_config_interfaces(driver, storage)


def test_get_network_instances(
    driver : GnmiOpenConfigDriver,  # pylint: disable=redefined-outer-name
    storage : Storage,              # pylint: disable=redefined-outer-name
) -> None:
    results_getconfig = check_config_network_instances(driver, storage)
    storage.network_instances.populate(results_getconfig)
    check_config_network_instances(driver, storage)


def test_set_network_instances(
    driver : GnmiOpenConfigDriver,  # pylint: disable=redefined-outer-name
    storage : Storage,              # pylint: disable=redefined-outer-name
) -> None:
    check_config_interfaces(driver, storage)
    check_config_network_instances(driver, storage)

    resources_to_set = list()
    ni_names = list()
    for ni in NETWORK_INSTANCES:
        ni_name = ni['name']
        ni_type = ni['type']
        resources_to_set.append(network_instance(ni_name, ni_type))
        ni_names.append(ni_name)
        storage.network_instances.network_instances.add(ni_name, {'type': ni_type})
    
    LOGGER.info('resources_to_set = {:s}'.format(str(resources_to_set)))
    results_setconfig = driver.SetConfig(resources_to_set)
    LOGGER.info('results_setconfig = {:s}'.format(str(results_setconfig)))
    check_updates(results_setconfig, '/network_instance[{:s}]', ni_names)

    check_config_interfaces(driver, storage, max_retries=5)
    check_config_network_instances(driver, storage, max_retries=5)


def test_set_interfaces(
    driver : GnmiOpenConfigDriver,  # pylint: disable=redefined-outer-name
    storage : Storage,              # pylint: disable=redefined-outer-name
) -> None:
    check_config_interfaces(driver, storage)
    check_config_network_instances(driver, storage)

    resources_to_set = list()
    if_names = list()
    for ni in NETWORK_INSTANCES:
        ni_name = ni['name']
        for ni_if in ni.get('interfaces', list()):
            if_name      = ni_if['if_name']
            subif_index  = ni_if['sif_index']
            ipv4_address = ni_if['ipv4_addr']
            ipv4_prefix  = ni_if['ipv4_prefix']
            enabled      = ni_if['enabled']
            resources_to_set.append(interface(
                if_name, subif_index, ipv4_address, ipv4_prefix, enabled
            ))
            if_names.append(ni_name)
            storage.interfaces.ipv4_addresses.add(if_name, subif_index, ipv4_address, {
                'origin' : 'STATIC', 'prefix': ipv4_prefix
            })

    LOGGER.info('resources_to_set = {:s}'.format(str(resources_to_set)))
    results_setconfig = driver.SetConfig(resources_to_set)
    LOGGER.info('results_setconfig = {:s}'.format(str(results_setconfig)))
    check_updates(results_setconfig, '/interface[{:s}]', if_names)

    check_config_interfaces(driver, storage, max_retries=5)
    check_config_network_instances(driver, storage, max_retries=5)


def test_add_interfaces_to_network_instance(
    driver : GnmiOpenConfigDriver,  # pylint: disable=redefined-outer-name
    storage : Storage,              # pylint: disable=redefined-outer-name
) -> None:
    check_config_interfaces(driver, storage)
    check_config_network_instances(driver, storage)

    resources_to_set = list()
    ni_if_names = list()
    for ni in NETWORK_INSTANCES:
        ni_name = ni['name']
        for ni_if in ni.get('interfaces', list()):
            if_name     = ni_if['if_name']
            subif_index = ni_if['sif_index']
            resources_to_set.append(network_instance_interface(ni_name, if_name, subif_index))
            ni_if_names.append((ni_name, '{:s}.{:d}'.format(if_name, subif_index)))
            storage.network_instances.interfaces.add(ni_name, if_name) # TODO: add subif_index

    LOGGER.info('resources_to_set = {:s}'.format(str(resources_to_set)))
    results_setconfig = driver.SetConfig(resources_to_set)
    LOGGER.info('results_setconfig = {:s}'.format(str(results_setconfig)))
    check_updates(results_setconfig, '/network_instance[{:s}]/interface[{:s}]', ni_if_names)

    check_config_interfaces(driver, storage, max_retries=5)
    check_config_network_instances(driver, storage, max_retries=5)


def test_set_network_instance_static_routes(
    driver : GnmiOpenConfigDriver,  # pylint: disable=redefined-outer-name
    storage : Storage,              # pylint: disable=redefined-outer-name
) -> None:
    check_config_interfaces(driver, storage)
    check_config_network_instances(driver, storage)

    resources_to_set = list()
    ni_sr_prefixes = list()
    for ni in NETWORK_INSTANCES:
        ni_name = ni['name']
        for ni_sr in ni.get('static_routes', list()):
            ni_sr_prefix  = ni_sr['prefix' ]
            ni_sr_gateway = ni_sr['gateway']
            ni_sr_metric  = ni_sr['metric' ]
            resources_to_set.append(
                network_instance_static_route(ni_name, ni_sr_prefix, ni_sr_gateway, metric=ni_sr_metric)
            )
            ni_sr_prefixes.append((ni_name, ni_sr_prefix))
            storage.network_instances.protocol_static.add(ni_name, 'STATIC', ni_sr_prefix, {
                'prefix': ni_sr_prefix, 
            })

    LOGGER.info('resources_to_set = {:s}'.format(str(resources_to_set)))
    results_setconfig = driver.SetConfig(resources_to_set)
    LOGGER.info('results_setconfig = {:s}'.format(str(results_setconfig)))
    check_updates(results_setconfig, '/network_instance[{:s}]/static_route[{:s}]', ni_sr_prefixes)

    check_config_interfaces(driver, storage, max_retries=5)
    check_config_network_instances(driver, storage, max_retries=5)


def test_del_network_instance_static_routes(
    driver : GnmiOpenConfigDriver,  # pylint: disable=redefined-outer-name
    storage : Storage,              # pylint: disable=redefined-outer-name
) -> None:
    check_config_interfaces(driver, storage)
    check_config_network_instances(driver, storage)

    resources_to_delete = list()
    ni_sr_prefixes = list()
    for ni in NETWORK_INSTANCES:
        ni_name = ni['name']
        for ni_sr in ni.get('static_routes', list()):
            ni_sr_prefix  = ni_sr['prefix' ]
            ni_sr_gateway = ni_sr['gateway']
            ni_sr_metric  = ni_sr['metric' ]
            resources_to_delete.append(
                network_instance_static_route(ni_name, ni_sr_prefix, ni_sr_gateway, metric=ni_sr_metric)
            )
            ni_sr_prefixes.append((ni_name, ni_sr_prefix))
            storage.network_instances.protocol_static.remove(ni_name, 'STATIC', ni_sr_prefix)

    LOGGER.info('resources_to_delete = {:s}'.format(str(resources_to_delete)))
    results_deleteconfig = driver.DeleteConfig(resources_to_delete)
    LOGGER.info('results_deleteconfig = {:s}'.format(str(results_deleteconfig)))
    check_updates(results_deleteconfig, '/network_instance[{:s}]/static_route[{:s}]', ni_sr_prefixes)

    check_config_interfaces(driver, storage, max_retries=5)
    check_config_network_instances(driver, storage, max_retries=5)


def test_del_interfaces_from_network_instance(
    driver : GnmiOpenConfigDriver,  # pylint: disable=redefined-outer-name
    storage : Storage,              # pylint: disable=redefined-outer-name
) -> None:
    check_config_interfaces(driver, storage)
    check_config_network_instances(driver, storage)

    resources_to_delete = list()
    ni_if_names = list()
    for ni in NETWORK_INSTANCES:
        ni_name = ni['name']
        for ni_if in ni.get('interfaces', list()):
            if_name     = ni_if['if_name']
            subif_index = ni_if['sif_index']
            resources_to_delete.append(network_instance_interface(ni_name, if_name, subif_index))
            ni_if_names.append((ni_name, '{:s}.{:d}'.format(if_name, subif_index)))
            storage.network_instances.interfaces.remove(ni_name, if_name) # TODO: add subif_index

    LOGGER.info('resources_to_delete = {:s}'.format(str(resources_to_delete)))
    results_deleteconfig = driver.DeleteConfig(resources_to_delete)
    LOGGER.info('results_deleteconfig = {:s}'.format(str(results_deleteconfig)))
    check_updates(results_deleteconfig, '/network_instance[{:s}]/interface[{:s}]', ni_if_names)
    
    check_config_interfaces(driver, storage, max_retries=5)
    check_config_network_instances(driver, storage, max_retries=5)


def test_del_interfaces(
    driver : GnmiOpenConfigDriver,  # pylint: disable=redefined-outer-name
    storage : Storage,              # pylint: disable=redefined-outer-name
) -> None:
    check_config_interfaces(driver, storage)
    check_config_network_instances(driver, storage)

    resources_to_delete = list()
    if_names = list()
    for ni in NETWORK_INSTANCES:
        ni_name = ni['name']
        for ni_if in ni.get('interfaces', list()):
            if_name      = ni_if['if_name']
            subif_index  = ni_if['sif_index']
            ipv4_address = ni_if['ipv4_addr']
            ipv4_prefix  = ni_if['ipv4_prefix']
            enabled      = ni_if['enabled']
            resources_to_delete.append(interface(if_name, subif_index, ipv4_address, ipv4_prefix, enabled))
            if_names.append(ni_name)
            storage.interfaces.ipv4_addresses.remove(if_name, subif_index, ipv4_address)

    LOGGER.info('resources_to_delete = {:s}'.format(str(resources_to_delete)))
    results_deleteconfig = driver.DeleteConfig(resources_to_delete)
    LOGGER.info('results_deleteconfig = {:s}'.format(str(results_deleteconfig)))
    check_updates(results_deleteconfig, '/interface[{:s}]', if_names)

    check_config_interfaces(driver, storage, max_retries=5)
    check_config_network_instances(driver, storage, max_retries=5)


def test_del_network_instances(
    driver : GnmiOpenConfigDriver,  # pylint: disable=redefined-outer-name
    storage : Storage,              # pylint: disable=redefined-outer-name
) -> None:
    check_config_interfaces(driver, storage)
    check_config_network_instances(driver, storage)

    resources_to_delete = list()
    ni_names = list()
    for ni in NETWORK_INSTANCES:
        ni_name = ni['name']
        ni_type = ni['type']
        resources_to_delete.append(network_instance(ni_name, ni_type))
        ni_names.append(ni_name)
        storage.network_instances.network_instances.remove(ni_name)

    LOGGER.info('resources_to_delete = {:s}'.format(str(resources_to_delete)))
    results_deleteconfig = driver.DeleteConfig(resources_to_delete)
    LOGGER.info('results_deleteconfig = {:s}'.format(str(results_deleteconfig)))
    check_updates(results_deleteconfig, '/network_instance[{:s}]', ni_names)

    check_config_interfaces(driver, storage, max_retries=5)
    check_config_network_instances(driver, storage, max_retries=5)
