# Copyright 2021-2023 H2020 TeraFlow (https://www.teraflow-h2020.eu/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from enum import IntEnum
from common.DeviceTypes import DeviceTypeEnum
from common.proto.context_pb2 import ServiceTypeEnum

class CapacityUnit(IntEnum):
    TB   = 0
    TBPS = 1
    GB   = 2
    GBPS = 3
    MB   = 4
    MBPS = 5
    KB   = 6
    KBPS = 7
    GHZ  = 8
    MHZ  = 9

CAPACITY_MULTIPLIER = {
    CapacityUnit.TB   : 1.e12,
    CapacityUnit.TBPS : 1.e12,
    CapacityUnit.GB   : 1.e9,
    CapacityUnit.GBPS : 1.e9,
    CapacityUnit.MB   : 1.e6,
    CapacityUnit.MBPS : 1.e6,
    CapacityUnit.KB   : 1.e3,
    CapacityUnit.KBPS : 1.e3,
    CapacityUnit.GHZ  : 1.e9,
    CapacityUnit.MHZ  : 1.e6,
}

class LinkPortDirection(IntEnum):
    BIDIRECTIONAL = 0
    INPUT         = 1
    OUTPUT        = 2
    UNKNOWN       = 3

class TerminationDirection(IntEnum):
    BIDIRECTIONAL = 0
    SINK          = 1
    SOURCE        = 2
    UNKNOWN       = 3

class TerminationState(IntEnum):
    CAN_NEVER_TERMINATE         = 0
    NOT_TERMINATED              = 1
    TERMINATED_SERVER_TO_CLIENT = 2
    TERMINATED_CLIENT_TO_SERVER = 3
    TERMINATED_BIDIRECTIONAL    = 4
    PERMENANTLY_TERMINATED      = 5
    TERMINATION_STATE_UNKNOWN   = 6

class LinkForwardingDirection(IntEnum):
    BIDIRECTIONAL  = 0
    UNIDIRECTIONAL = 1
    UNKNOWN        = 2

class DeviceLayerEnum(IntEnum):
    APPLICATION_CONTROLLER = 41     # Layer 4 domain controller
    APPLICATION_DEVICE     = 40     # Layer 4 domain device
    PACKET_CONTROLLER      = 31     # Layer 3 domain controller
    PACKET_DEVICE          = 30     # Layer 3 domain device
    MAC_LAYER_CONTROLLER   = 21     # Layer 2 domain controller
    MAC_LAYER_DEVICE       = 20     # Layer 2 domain device
    OPTICAL_CONTROLLER     =  1     # Layer 0 domain controller
    OPTICAL_DEVICE         =  0     # Layer 0 domain device

DEVICE_TYPE_TO_LAYER = {
    DeviceTypeEnum.EMULATED_DATACENTER.value      : DeviceLayerEnum.APPLICATION_DEVICE,
    DeviceTypeEnum.DATACENTER.value               : DeviceLayerEnum.APPLICATION_DEVICE,
    DeviceTypeEnum.NETWORK.value                  : DeviceLayerEnum.APPLICATION_DEVICE,

    DeviceTypeEnum.EMULATED_PACKET_ROUTER.value   : DeviceLayerEnum.PACKET_DEVICE,
    DeviceTypeEnum.PACKET_ROUTER.value            : DeviceLayerEnum.PACKET_DEVICE,
    DeviceTypeEnum.EMULATED_PACKET_SWITCH.value   : DeviceLayerEnum.MAC_LAYER_DEVICE,
    DeviceTypeEnum.PACKET_SWITCH.value            : DeviceLayerEnum.MAC_LAYER_DEVICE,
    DeviceTypeEnum.P4_SWITCH.value                : DeviceLayerEnum.MAC_LAYER_DEVICE,

    DeviceTypeEnum.MICROVAWE_RADIO_SYSTEM.value   : DeviceLayerEnum.MAC_LAYER_CONTROLLER,

    DeviceTypeEnum.EMULATED_OPEN_LINE_SYSTEM.value: DeviceLayerEnum.OPTICAL_CONTROLLER,
    DeviceTypeEnum.OPEN_LINE_SYSTEM.value         : DeviceLayerEnum.OPTICAL_CONTROLLER,

    DeviceTypeEnum.OPTICAL_ROADM.value            : DeviceLayerEnum.OPTICAL_DEVICE,
    DeviceTypeEnum.OPTICAL_TRANSPONDER.value      : DeviceLayerEnum.OPTICAL_DEVICE,
}

DEVICE_LAYER_TO_SERVICE_TYPE = {
    DeviceLayerEnum.APPLICATION_DEVICE.value: ServiceTypeEnum.SERVICETYPE_L3NM,

    DeviceLayerEnum.PACKET_DEVICE.value     : ServiceTypeEnum.SERVICETYPE_L3NM,
    DeviceLayerEnum.MAC_LAYER_DEVICE.value  : ServiceTypeEnum.SERVICETYPE_L2NM,

    DeviceLayerEnum.OPTICAL_CONTROLLER.value: ServiceTypeEnum.SERVICETYPE_TAPI_CONNECTIVITY_SERVICE,
}
