# Copyright 2022-2024 ETSI OSG/SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import grpc, logging
from common.method_wrappers.Decorator import MetricsPool, safe_and_metered_rpc_method

from common.proto.resource_manager_pb2_grpc import ResourceManagerServicer
from common.proto.resource_manager_pb2 import (
    L2Request, L2Response,
    L3Request, L3Response
)

# Métricas
RESOURCE_POOL = MetricsPool('ResourceManager', 'RPC')
LOGGER = logging.getLogger(__name__)


class ResourceManagerServicerImpl(ResourceManagerServicer):
    def __init__(self):
        LOGGER.info("Initializing ResourceManagerServicerImpl")

        self.assigned_vlans = {}  # (device_uuid, endpoint_uuid) -> vlan_id
        self.assigned_ips = {}    # (device_uuid, endpoint_uuid) -> ip_address

    @safe_and_metered_rpc_method(RESOURCE_POOL, LOGGER)
    def VerifyAndAssignL2(self, request: L2Request, context: grpc.ServicerContext) -> L2Response:
        key = (request.device_uuid, request.endpoint_uuid)
        LOGGER.info("Vlanid: %s, Endpoint: %s", request.vlan_id, key)
        if key in self.assigned_vlans:
            assigned_vlan = self.assigned_vlans[key]
            if assigned_vlan == request.vlan_id:
                msg = f"VLAN {request.vlan_id} already assigned to endpoint {key}."
                LOGGER.info(msg)
                return L2Response(success=True, message=msg)
            else:
                msg = f"Conflict: endpoint {key} already has VLAN {assigned_vlan} assigned."
                LOGGER.warning(msg)
                return L2Response(success=False, message=msg)
        else:
            self.assigned_vlans[key] = request.vlan_id
            msg = f"Assigned VLAN {request.vlan_id} to endpoint {key}."
            LOGGER.info(msg)
            return L2Response(success=True, message=msg)
        

    @safe_and_metered_rpc_method(RESOURCE_POOL, LOGGER)
    def VerifyAndAssignL3(self, request: L3Request, context: grpc.ServicerContext) -> L3Response:
        key = (request.device_uuid, request.endpoint_uuid)

        if key in self.assigned_ips:
            assigned_ip = self.assigned_ips[key]
            if assigned_ip == request.ip_address:
                msg = f"IP {request.ip_address} already assigned to endpoint {key}."
                LOGGER.info(msg)
                return L3Response(success=True, message=msg)
            else:
                msg = f"Conflict: endpoint {key} already has IP {assigned_ip} assigned."
                LOGGER.warning(msg)
                return L3Response(success=False, message=msg)
        else:
            self.assigned_ips[key] = request.ip_address
            msg = f"Assigned IP {request.ip_address} to endpoint {key}."
            LOGGER.info(msg)
            return L3Response(success=True, message=msg)