# Copyright 2022-2023 ETSI TeraFlowSDN - TFS OSG (https://tfs.etsi.org/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import grpc, logging
from common.Constants import ServiceNameEnum
from common.Settings import get_service_host, get_service_port_grpc
from common.proto.context_pb2 import Empty, Service, ServiceId
from common.proto.pcep_pb2_grpc import PcepServiceStub
from common.proto.pcep_pb2 import RequestRq, RequestRp, PceIpRq, PceIpRp, LSPdb_Response, LSPdb_Request, commandResponse
from common.tools.client.RetryDecorator import retry, delay_exponential
from common.tools.grpc.Tools import grpc_message_to_json_string

from pcep.service.tools.protos.grpcService_pb2 import  Session_Request, commandRequest
from pcep.service.tools.protos.grpcService_pb2_grpc import pceServiceStub
from pcep.database.LSP_DB import LspDB



LOGGER = logging.getLogger(__name__)
MAX_RETRIES = 15
DELAY_FUNCTION = delay_exponential(initial=0.01, increment=2.0, maximum=5.0)
RETRY_DECORATOR = retry(max_retries=MAX_RETRIES, delay_function=DELAY_FUNCTION, prepare_method_name='connect')

PORT = 10060

class PcepClient:
    def __init__(self, host=None, port=None):
        if not host: host = get_service_host(ServiceNameEnum.PCEP)
        
        # if not host: host = PcepClient.get_loadbalancer_ip('pcepservice')
        
        if not port: port = get_service_port_grpc(ServiceNameEnum.PCEP)
        
        # if not port: port = port = PcepClient.get_loadbalancer_port('pcepservice', namespace='tfs', target_port_name='grpc')
        
        self.endpoint = '{:s}:{:s}'.format(str(host), str(port))
        LOGGER.info('Creating channel to {:s}...'.format(str(self.endpoint)))
        self.channel = None
        self.stub = None
        self.connect()
        LOGGER.info('Channel created')
        
        # # # Inicializamos LspDB para acceder a la base de datos
        # self.lsp_db = LspDB()
        # self.lsp_db.create_database()
        # self.lsp_db.create_tables()
        # self.lsp_db.verify_tables()

    def connect(self):
        LOGGER.debug('Creating channel to {:s}...'.format(str(self.endpoint)))
        self.channel = grpc.insecure_channel(self.endpoint)
        self.stub = PcepServiceStub(self.channel)
        self.host = get_service_host(ServiceNameEnum.PCEP)
        LOGGER.debug('Channel created')
        
    def close(self):
        if self.channel is not None: self.channel.close()
        self.channel = None
        self.stub = None

    
    


    @RETRY_DECORATOR
    def sendRequest(self, request : RequestRq) -> commandResponse:
        LOGGER.debug('Send request: {:s}'.format(grpc_message_to_json_string(request)))
        response = self.stub.sendRequest(request)
        LOGGER.debug('Send request result: {:s}'.format(grpc_message_to_json_string(response)))
        return response

    @RETRY_DECORATOR
    def configuratePCE(self, request : PceIpRq) -> PceIpRp:
        LOGGER.debug("Configuring the PCE")
        LOGGER.debug('Configurate PCE: {:s}'.format(grpc_message_to_json_string(request)))
        response = self.stub.configuratePCE(request)
        LOGGER.debug('Configurate PCE result: {:s}'.format(grpc_message_to_json_string(response)))
        return response
    
    
    # @RETRY_DECORATOR
    # def showLSPDB(self):
    #     LOGGER.debug("Showing the LSP Paths in the PCE")
    #     requestlsp = LSPdb_Request()
    #     channel = grpc.insecure_channel(f'{self.host}:{PORT}')
    #     stub_lsp = pceServiceStub(channel)
    #     responselsp = stub_lsp.getLSPdb(requestlsp)
    #     # responselsp = self.stub.getLSPdb(request)
    #     LOGGER.warning("IAM IN SHOWLSP: {:s}".format(str(responselsp)))
    #     return responselsp

    @RETRY_DECORATOR
    def showLSPDB(self):
        LOGGER.debug("Showing the LSP Paths from the database")
        # Llamamos a LspDB para obtener la lista de LSPs
        try:
            lsp_info = LspDB.show_lsp_db()
            LOGGER.info(f"Información de LSPs: {lsp_info}")
            return lsp_info
        except Exception as e:
            LOGGER.error(f"Error al obtener la información de LSP desde la base de datos: {str(e)}")
            return None

    #EL BUENO  
    # @RETRY_DECORATOR
    # def showLSPDB(self):
    #     LOGGER.debug("Showing the LSP Paths in the PCE")
    #     requestlsp = LSPdb_Request()
    #     responselsp = self.stub.getLSPdb(requestlsp)
    #     LOGGER.warning("IAM IN SHOWLSP: {:s}".format(str(responselsp)))
    #     return responselsp
    
    @RETRY_DECORATOR
    def showSessions(self):
        requestsessions = Session_Request()
        # channel = grpc.insecure_channel(f'{self.host}:{PORT}')
        # stub_sessions = pceServiceStub(channel)
        responsesessions = self.stub.getSessionsInfo(requestsessions)
        LOGGER.warning("IAM IN SHOWSESSIONS: {:s}".format(str(responsesessions)))
        return responsesessions
    
    @RETRY_DECORATOR
    def sendUpdate(self, data):
        LOGGER.warning("IAM IN SEND UPDATE")
        commandrequest = commandRequest()  # Create an instance without passing data to __init__
        commandrequest.command = data
        
        LOGGER.warning("This is Command Request: %s", str(commandrequest))
        
        channel = grpc.insecure_channel(f'{self.host}:{PORT}')
        stub_update = pceServiceStub(channel)
        commandresponse = stub_update.update(commandrequest)
        
        LOGGER.warning("This is Command Response: %s", str(commandresponse))
        
        LOGGER.warning("Success: %s", str(commandresponse.success))
        LOGGER.warning("Error Message: %s", str(commandresponse.error_message))
        return commandresponse
    
