from __future__ import annotations
from typing import TYPE_CHECKING, Dict, List
from .._engine.object._Object import _Object
from .._engine.object.Attributes import Attributes
from .._engine.object.Collection import Collection
from ._Keys import KEY_TOPOLOGY, KEY_DEVICES, KEY_LINKS
from .Device import Device
from .Link import Link

if TYPE_CHECKING:
    from .Context import Context

VALIDATORS = {}  # no attributes accepted
TRANSCODERS = {} # no transcoding applied to attributes

class Topology(_Object):
    def __init__(self, topology_uuid : str, parent : 'Context'):
        super().__init__(parent, topology_uuid)
        self._attributes = Attributes(parent, KEY_TOPOLOGY, VALIDATORS, TRANSCODERS)
        self._devices = Collection(self, KEY_DEVICES)
        self._links = Collection(self, KEY_LINKS)

    @property
    def parent(self) -> 'Context': return self._parent

    @property
    def context(self) -> 'Context': return self._parent

    @property
    def context_uuid(self) -> str: return self.context.context_uuid

    @property
    def topology_uuid(self) -> str: return self._object_uuid

    @property
    def attributes(self) -> Attributes: return self._attributes

    @property
    def devices(self) -> Collection: return self._devices

    @property
    def links(self) -> Collection: return self._links

    def device(self, device_uuid : str) -> Device: return Device(device_uuid, self)

    def link(self, link_uuid : str) -> Link: return Link(link_uuid, self)

    def create(self) -> 'Topology':
        self.parent.topologies.add(self.topology_uuid)
        return self

    def update(self, update_attributes={}, remove_attributes=[]) -> 'Topology':
        self.attributes.update(update_attributes=update_attributes, remove_attributes=remove_attributes)
        return self

    def delete(self) -> None:
        for device_uuid in self.devices.get(): self.device(device_uuid).delete()
        for link_uuid in self.links.get(): self.link(link_uuid).delete()
        self.attributes.delete()
        self.parent.topologies.delete(self.topology_uuid)

    def dump_id(self) -> Dict:
        return {
            'context_id': self.context.dump_id(),
            'topology_uuid': {'uuid': self.topology_uuid},
        }

    def dump_devices(self) -> List[Dict]:
        return [self.device(device_uuid).dump() for device_uuid in self.devices.get()]

    def dump_links(self) -> List[Dict]:
        return [self.link(link_uuid).dump() for link_uuid in self.links.get()]

    def dump(self, include_devices=True, include_links=True) -> Dict:
        result = {'topology_id': self.dump_id()}
        if include_devices: result['devices'] = self.dump_devices()
        if include_links: result['links'] = self.dump_links()
        return result
