from __future__ import annotations
from typing import TYPE_CHECKING, Dict, List
from .._engine.object._Object import _Object
from .._engine.object.Attributes import Attributes
from .._engine.object.Collection import Collection
from ._Keys import KEY_CONTEXT, KEY_SERVICES, KEY_TOPOLOGIES
from .Service import Service
from .Topology import Topology

if TYPE_CHECKING:
    from ..Database import Database

VALIDATORS = {}  # no attributes accepted
TRANSCODERS = {} # no transcoding applied to attributes

class Context(_Object):
    def __init__(self, context_uuid : str, parent : 'Database'):
        super().__init__(parent, context_uuid)
        self._attributes = Attributes(parent, KEY_CONTEXT, VALIDATORS, TRANSCODERS)
        self._services = Collection(self, KEY_SERVICES)
        self._topologies = Collection(self, KEY_TOPOLOGIES)

    @property
    def parent(self) -> 'Database': return self._parent

    @property
    def context(self) -> 'Context': return self

    @property
    def context_uuid(self) -> str: return self._object_uuid

    @property
    def attributes(self) -> Attributes: return self._attributes

    @property
    def services(self) -> Collection: return self._services

    @property
    def topologies(self) -> Collection: return self._topologies

    def service(self, service_uuid : str) -> Service: return Service(service_uuid, self)

    def topology(self, topology_uuid : str) -> Topology: return Topology(topology_uuid, self)

    def create(self) -> 'Context':
        self.parent.contexts.add(self.context_uuid)
        return self

    def update(self, update_attributes={}, remove_attributes=[]) -> 'Context':
        self.attributes.update(update_attributes=update_attributes, remove_attributes=remove_attributes)
        return self

    def delete(self):
        for service_uuid in self.services.get(): self.service(service_uuid).delete()
        for topology_uuid in self.topologies.get(): self.topology(topology_uuid).delete()
        self.attributes.delete()
        self.parent.contexts.delete(self.context_uuid)

    def dump_id(self) -> Dict:
        return {'context_uuid': {'uuid': self.context_uuid}}

    def dump_services(self) -> List[Dict]:
        return [self.service(service_uuid).dump() for service_uuid in self.services.get()]

    def dump_topologies(self) -> List[Dict]:
        return [self.topology(topology_uuid).dump() for topology_uuid in self.topologies.get()]

    def dump(self, include_services=True, include_topologies=True) -> Dict:
        result = {'context_id': self.dump_id()}
        if include_services: result['services'] = self.dump_services()
        if include_topologies: result['topologies'] = self.dump_topologies()
        return result
