package eu.teraflow.automation;

import static org.assertj.core.api.AssertionsForClassTypes.assertThat;

import automation.Automation;
import context.ContextOuterClass;
import eu.teraflow.automation.context.ContextGateway;
import eu.teraflow.automation.device.Device;
import eu.teraflow.automation.device.DeviceGateway;
import eu.teraflow.automation.device.model.*;
import eu.teraflow.automation.model.DeviceRole;
import eu.teraflow.automation.model.DeviceRoleId;
import eu.teraflow.automation.model.DeviceRoleType;
import io.quarkus.test.junit.QuarkusTest;
import io.quarkus.test.junit.mockito.InjectMock;
import io.smallrye.mutiny.Uni;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import javax.inject.Inject;
import org.assertj.core.api.Assertions;
import org.jboss.logging.Logger;
import org.junit.jupiter.api.Test;
import org.mockito.Mockito;

@QuarkusTest
public class AutomationFunctionalServiceTest {
    private static final Logger LOGGER = Logger.getLogger(AutomationFunctionalServiceTest.class);

    @Inject AutomationService automationService;

    @InjectMock DeviceGateway deviceGateway;
    @InjectMock ContextGateway contextGateway;

    @Test
    void shouldConfigureDevice() {

        final var uuidForDeviceRoleId =
                ContextOuterClass.Uuid.newBuilder()
                        .setUuid(UUID.fromString("0f14d0ab-9608-7862-a9e4-5ed26688389b").toString())
                        .build();

        final var uuidForDeviceId =
                ContextOuterClass.Uuid.newBuilder()
                        .setUuid(UUID.fromString("9f14d0ab-9608-7862-a9e4-5ed26688389c").toString())
                        .build();

        final var outDeviceId =
                ContextOuterClass.DeviceId.newBuilder().setDeviceUuid(uuidForDeviceId).build();

        final var outDeviceRoleId =
                Automation.DeviceRoleId.newBuilder()
                        .setDevRoleId(uuidForDeviceRoleId)
                        .setDevId(outDeviceId)
                        .build();

        Uuid uuid = new Uuid(outDeviceRoleId.getDevId().toString());
        DeviceId deviceId = new DeviceId(uuid);
        DeviceType deviceType = new DeviceType("cisco");

        ConfigRule configRule1 = new ConfigRule(ConfigActionEnum.UNDEFINED, "001", "100");
        ConfigRule configRule2 = new ConfigRule(ConfigActionEnum.SET, "002", "101");
        List<ConfigRule> configRuleList = new ArrayList<>();
        configRuleList.add(configRule1);
        configRuleList.add(configRule2);

        DeviceConfig expectedDeviceConfig = new DeviceConfig(configRuleList);
        Uni<DeviceConfig> expectedDeviceConfigUni = Uni.createFrom().item(expectedDeviceConfig);
        Uni<DeviceId> expectedDeviceId = Uni.createFrom().item(deviceId);

        DeviceRoleId deviceRoleId = new DeviceRoleId(uuid, deviceId);

        DeviceRole deviceRole = new DeviceRole(deviceRoleId, DeviceRoleType.DEV_CONF);

        Device device = new Device(deviceId, deviceType, DeviceOperationalStatus.DISABLED);
        Uni<Device> deviceUni = Uni.createFrom().item(device);

        Mockito.when(contextGateway.getDevice(Mockito.any())).thenReturn(deviceUni);
        Mockito.when(deviceGateway.getInitialConfiguration(Mockito.any()))
                .thenReturn(expectedDeviceConfigUni);
        Mockito.when(deviceGateway.configureDevice(Mockito.any())).thenReturn(expectedDeviceId);

        final var currentDevice = automationService.ztpAdd(deviceRole);

        Assertions.assertThat(currentDevice).isNotNull();
        currentDevice
                .subscribe()
                .with(
                        deviceConfig -> {
                            LOGGER.infof("Received response %s", deviceConfig);
                            assertThat(deviceConfig.getDeviceOperationalStatus().toString())
                                    .isEqualTo(device.getDeviceOperationalStatus().toString());
                            assertThat(deviceConfig.getDeviceConfig().toString()).isNotNull();
                            assertThat(deviceConfig.getDeviceConfig().toString())
                                    .isEqualTo(expectedDeviceConfig.toString());
                            assertThat(deviceConfig.getDeviceId().getId()).isEqualTo(deviceId.getId());
                        });
    }

    @Test
    void shouldNotConfigureDevice() {

        final var uuidForDeviceRoleId =
                ContextOuterClass.Uuid.newBuilder()
                        .setUuid(UUID.fromString("2f14d0ab-9608-7862-a9e4-5ed26688389f").toString())
                        .build();

        final var uuidForDeviceId =
                ContextOuterClass.Uuid.newBuilder()
                        .setUuid(UUID.fromString("3f14d0ab-9608-7862-a9e4-5ed26688389d").toString())
                        .build();

        final var outDeviceId =
                ContextOuterClass.DeviceId.newBuilder().setDeviceUuid(uuidForDeviceId).build();

        final var outDeviceRoleId =
                Automation.DeviceRoleId.newBuilder()
                        .setDevRoleId(uuidForDeviceRoleId)
                        .setDevId(outDeviceId)
                        .build();

        Uuid uuid = new Uuid(outDeviceRoleId.getDevId().toString());
        DeviceId deviceId = new DeviceId(uuid);
        DeviceType deviceType = new DeviceType("ztp");

        List<ConfigRule> configRuleList = new ArrayList<>();

        ConfigRule configRule1 =
                new ConfigRule(ConfigActionEnum.UNDEFINED, "001", "already-configured");
        configRuleList.add(configRule1);

        DeviceConfig expectedDeviceConfig = new DeviceConfig(configRuleList);

        Uuid devRoleIdUuid = new Uuid(outDeviceRoleId.getDevRoleId().getUuid());
        DeviceRoleId deviceRoleId = new DeviceRoleId(devRoleIdUuid, deviceId);

        DeviceRole deviceRole = new DeviceRole(deviceRoleId, DeviceRoleType.PIPELINE_CONF);

        Device device =
                new Device(deviceId, deviceType, expectedDeviceConfig, DeviceOperationalStatus.ENABLED);

        Uni<Device> deviceUni = Uni.createFrom().item(device);
        Mockito.when(contextGateway.getDevice(Mockito.any())).thenReturn(deviceUni);

        final var currentDevice = automationService.ztpAdd(deviceRole);

        Assertions.assertThat(currentDevice).isNotNull();

        currentDevice
                .subscribe()
                .with(
                        deviceConfig -> {
                            LOGGER.infof("Received response %s", deviceConfig);
                            assertThat(deviceConfig.getDeviceOperationalStatus().toString())
                                    .isEqualTo(device.getDeviceOperationalStatus().toString());
                            assertThat(deviceConfig.getDeviceConfig().toString()).isNotNull();
                            assertThat(deviceConfig.getDeviceConfig().toString())
                                    .isEqualTo(expectedDeviceConfig.toString());
                            assertThat(deviceConfig.getDeviceId().getId().toString())
                                    .isEqualTo(deviceId.getId().toString());
                        });
    }
}
