# Copyright 2022-2023 ETSI TeraFlowSDN - TFS OSG (https://tfs.etsi.org/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from flask.json import jsonify
from flask_restful import Resource
from common.proto.context_pb2 import ConnectionId, ContextId, DeviceId, Empty, LinkId, ServiceId, SliceId, TopologyId
from common.proto.policy_pb2 import PolicyRuleId
from common.tools.grpc.Tools import grpc_message_to_json
from common.tools.object_factory.Connection import json_connection_id
from common.tools.object_factory.Context import json_context_id
from common.tools.object_factory.Device import json_device_id
from common.tools.object_factory.Link import json_link_id
from common.tools.object_factory.PolicyRule import json_policyrule_id
from common.tools.object_factory.Service import json_service_id
from common.tools.object_factory.Slice import json_slice_id
from common.tools.object_factory.Topology import json_topology_id
from context.client.ContextClient import ContextClient


def format_grpc_to_json(grpc_reply):
    return jsonify(grpc_message_to_json(grpc_reply))

def grpc_connection_id(connection_uuid):
    return ConnectionId(**json_connection_id(connection_uuid))

def grpc_context_id(context_uuid):
    return ContextId(**json_context_id(context_uuid))

def grpc_device_id(device_uuid):
    return DeviceId(**json_device_id(device_uuid))

def grpc_link_id(link_uuid):
    return LinkId(**json_link_id(link_uuid))

def grpc_service_id(context_uuid, service_uuid):
    return ServiceId(**json_service_id(service_uuid, context_id=json_context_id(context_uuid)))

def grpc_slice_id(context_uuid, slice_uuid):
    return SliceId(**json_slice_id(slice_uuid, context_id=json_context_id(context_uuid)))
    
def grpc_topology_id(context_uuid, topology_uuid):
    return TopologyId(**json_topology_id(topology_uuid, context_id=json_context_id(context_uuid)))

def grpc_policy_rule_id(policy_rule_uuid):
    return PolicyRuleId(**json_policyrule_id(policy_rule_uuid))


class _Resource(Resource):
    def __init__(self) -> None:
        super().__init__()
        self.client = ContextClient()

class ContextIds(_Resource):
    def get(self):
        return format_grpc_to_json(self.client.ListContextIds(Empty()))

class Contexts(_Resource):
    def get(self):
        return format_grpc_to_json(self.client.ListContexts(Empty()))

class Context(_Resource):
    def get(self, context_uuid : str):
        return format_grpc_to_json(self.client.GetContext(grpc_context_id(context_uuid)))

class TopologyIds(_Resource):
    def get(self, context_uuid : str):
        return format_grpc_to_json(self.client.ListTopologyIds(grpc_context_id(context_uuid)))

class Topologies(_Resource):
    def get(self, context_uuid : str):
        return format_grpc_to_json(self.client.ListTopologies(grpc_context_id(context_uuid)))

class Topology(_Resource):
    def get(self, context_uuid : str, topology_uuid : str):
        return format_grpc_to_json(self.client.GetTopology(grpc_topology_id(context_uuid, topology_uuid)))

class ServiceIds(_Resource):
    def get(self, context_uuid : str):
        return format_grpc_to_json(self.client.ListServiceIds(grpc_context_id(context_uuid)))

class Services(_Resource):
    def get(self, context_uuid : str):
        return format_grpc_to_json(self.client.ListServices(grpc_context_id(context_uuid)))

class Service(_Resource):
    def get(self, context_uuid : str, service_uuid : str):
        return format_grpc_to_json(self.client.GetService(grpc_service_id(context_uuid, service_uuid)))

class SliceIds(_Resource):
    def get(self, context_uuid : str):
        return format_grpc_to_json(self.client.ListSliceIds(grpc_context_id(context_uuid)))

class Slices(_Resource):
    def get(self, context_uuid : str):
        return format_grpc_to_json(self.client.ListSlices(grpc_context_id(context_uuid)))

class Slice(_Resource):
    def get(self, context_uuid : str, slice_uuid : str):
        return format_grpc_to_json(self.client.GetSlice(grpc_slice_id(context_uuid, slice_uuid)))

class DeviceIds(_Resource):
    def get(self):
        return format_grpc_to_json(self.client.ListDeviceIds(Empty()))

class Devices(_Resource):
    def get(self):
        return format_grpc_to_json(self.client.ListDevices(Empty()))

class Device(_Resource):
    def get(self, device_uuid : str):
        return format_grpc_to_json(self.client.GetDevice(grpc_device_id(device_uuid)))

class LinkIds(_Resource):
    def get(self):
        return format_grpc_to_json(self.client.ListLinkIds(Empty()))

class Links(_Resource):
    def get(self):
        return format_grpc_to_json(self.client.ListLinks(Empty()))

class Link(_Resource):
    def get(self, link_uuid : str):
        return format_grpc_to_json(self.client.GetLink(grpc_link_id(link_uuid)))

class ConnectionIds(_Resource):
    def get(self, context_uuid : str, service_uuid : str):
        return format_grpc_to_json(self.client.ListConnectionIds(grpc_service_id(context_uuid, service_uuid)))

class Connections(_Resource):
    def get(self, context_uuid : str, service_uuid : str):
        return format_grpc_to_json(self.client.ListConnections(grpc_service_id(context_uuid, service_uuid)))

class Connection(_Resource):
    def get(self, connection_uuid : str):
        return format_grpc_to_json(self.client.GetConnection(grpc_connection_id(connection_uuid)))

class PolicyRuleIds(_Resource):
    def get(self):
        return format_grpc_to_json(self.client.ListPolicyRuleIds(Empty()))

class PolicyRules(_Resource):
    def get(self):
        return format_grpc_to_json(self.client.ListPolicyRules(Empty()))

class PolicyRule(_Resource):
    def get(self, policy_rule_uuid : str):
        return format_grpc_to_json(self.client.GetPolicyRule(grpc_policy_rule_id(policy_rule_uuid)))
