from __future__ import annotations
from typing import TYPE_CHECKING, Dict
from ...entity._Entity import _Entity
from ...entity.EntityCollection import EntityCollection
from ..Keys import KEY_TOPOLOGY, KEY_DEVICES, KEY_LINKS
from .device.Device import Device
from .link.Link import Link

if TYPE_CHECKING:
    from ..Context import Context

VALIDATORS = {}  # no attributes accepted
TRANSCODERS = {} # no transcoding applied to attributes

class Topology(_Entity):
    def __init__(self, topology_uuid : str, parent : 'Context'):
        super().__init__(parent, topology_uuid, KEY_TOPOLOGY, VALIDATORS, TRANSCODERS)
        self._devices = EntityCollection(self, KEY_DEVICES)
        self._links = EntityCollection(self, KEY_LINKS)

    @property
    def parent(self) -> 'Context': return self._parent

    @property
    def context(self) -> 'Context': return self._parent

    @property
    def context_uuid(self) -> str: return self.context.context_uuid

    @property
    def topology_uuid(self) -> str: return self._entity_uuid

    @property
    def devices(self) -> EntityCollection: return self._devices

    @property
    def links(self) -> EntityCollection: return self._links

    def device(self, device_uuid : str) -> Device: return Device(device_uuid, self)

    def link(self, link_uuid : str) -> Link: return Link(link_uuid, self)

    def create(self) -> 'Topology':
        self.parent.topologies.add(self.topology_uuid)
        return self

    def delete(self) -> None:
        for device_uuid in self.devices.get(): self.device(device_uuid).delete()
        for link_uuid in self.links.get(): self.link(link_uuid).delete()
        self.attributes.delete()
        self.parent.topologies.delete(self.topology_uuid)

    def dump_id(self) -> Dict:
        return {
            'contextId': self.context.dump_id(),
            'topoId': {'uuid': self.topology_uuid},
        }

    def dump(self) -> Dict:
        devices = [self.device(device_uuid).dump() for device_uuid in self.devices.get()]
        links = [self.link(link_uuid).dump() for link_uuid in self.links.get()]
        return {
            'topoId': self.dump_id(),
            'device': devices,
            'link': links,
        }
