# Copyright 2022-2024 ETSI OSG/SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import libyang, os
from common.proto.context_pb2 import Device
from typing import Dict, Optional
import json
import logging
import re
import datetime

LOGGER = logging.getLogger(__name__)
YANG_DIR = os.path.join(os.path.dirname(__file__), 'yang')
YANG_MODULES = [
    'iana-hardware',
    'ietf-hardware'
]

class YangHandler:
    def __init__(self) -> None:
        self._yang_context = libyang.Context(YANG_DIR)
        for yang_module_name in YANG_MODULES:
            LOGGER.info('Loading module: {:s}'.format(str(yang_module_name)))
            self._yang_context.load_module(yang_module_name).feature_enable_all()

    def parse_to_dict(self, message : Dict) -> Dict:
        yang_module = self._yang_context.get_module('ietf-hardware')
        dnode : Optional[libyang.DNode] = yang_module.parse_data_dict(
            message, validate_present=True, validate=True, strict=True
        )
        if dnode is None: raise Exception('Unable to parse Message({:s})'.format(str(message)))
        message = dnode.print_dict()
        dnode.free()
        return message

    
    @staticmethod
    def convert_to_iso_date(date_str: str) -> Optional[str]:
        date_str = date_str.strip('"')
        # Define the regex pattern for ISO 8601 date format
        pattern = r"\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}(\.\d+)?(Z|[\+\-]\d{2}:\d{2})"
        # Check if the input date string matches the pattern
        if re.match(pattern, date_str):
            return date_str  # Already in ISO format
        else:
            try:
                # Parse the input date string as a datetime object
                datetime_obj = datetime.datetime.strptime(date_str, "%Y-%m-%d")
                # Convert to ISO format
                iso_date = datetime_obj.isoformat() + "Z"
                return iso_date
            except ValueError:
                return None  # Invalid date format


    def compose(self, device : Device) -> Dict:
        # compose device iterating through the components
 
        hardware = self._yang_context.create_data_path('/ietf-hardware:hardware')
        physical_index = 1
        
        for component in device.components:
            attributes = component.attributes

            component_new = hardware.create_path('component[name="{:s}"]'.format(component.name))
            component_new.create_path('name', component.name)

            #Cambiar las clases especiales, su formato  y añadir isfru 
            component_type = component.type
            if component_type == "TRANSCEIVER" :
                component_type = "module"

            if component_type == "FRU" :
                component_type = "slack"
                component_new.create_path('is-fru', True)
            else :
                component_new.create_path('is-fru', False)
                
            component_type = component_type.replace("_", "-").lower()
            component_type = 'iana-hardware:' + component_type

            component_new.create_path('class', component_type)

            #Añadir resto de atributos en IETF

            physical_index += 1
            component_new.create_path('physical-index', physical_index)

            component_new.create_path('description', attributes["description"])

            component_new.create_path('parent', component.parent)

            if attributes["mfg-date"] != "":
                mfg_date = self.convert_to_iso_date(attributes["mfg-date"])
                LOGGER.info('component[name="{:s}"]'.format(attributes["mfg-date"]))
                component_new.create_path('mfg-date', mfg_date)

            component_new.create_path('hardware-rev', attributes["hardware-rev"])
            component_new.create_path('software-rev', attributes["software-rev"])
            component_new.create_path('firmware-rev', attributes["firmware-version"])
            component_new.create_path('serial-num', attributes["serial-num"])
            component_new.create_path('mfg-name', attributes["mfg-name"])
            if attributes["id"]:
                component_new.create_path('parent-rel-pos', attributes["id"])
            
            component_new.create_path('uri', component.name)
   

            component_new.create_path('uuid', component.component_uuid.uuid)

            contains_child = []
            for component2 in device.components:
                if component.name == component2.parent : 
                 contains_child.append(component2.name)
            
            component_new.create_path('contains-child', contains_child)

        return json.loads(hardware.print_mem('json'))
                                    
    def destroy(self) -> None:
        self._yang_context.destroy()