import grpc
from common.proto.context_pb2 import Empty, ContextId, TopologyId, Uuid
from common.proto.context_pb2_grpc import ContextServiceStub
import unittest
from unittest.mock import patch, MagicMock

def get_context_topology_info():
    # Establish a gRPC channel
    channel = grpc.insecure_channel('10.152.183.77:1010')  # Update with the correct IP and port
    stub = ContextServiceStub(channel)

    # Retrieve the context information
    context_list = stub.ListContexts(Empty())
    contexts_info = []
    for context in context_list.contexts:
        context_info = {
            'context_id': context.context_id.context_uuid.uuid,
            'context_name': context.name,
            'topologies': []
        }

        # Retrieve topology information for each context
        topology_list = stub.ListTopologies(context.context_id)
        for topology in topology_list.topologies:
            topology_info = {
                'topology_id': topology.topology_id.topology_uuid.uuid,
                'topology_name': topology.name,
                'devices': []
            }

            # Retrieve detailed topology information
            topology_details = stub.GetTopologyDetails(topology.topology_id)
            for device in topology_details.devices:
                device_info = {
                    'device_id': device.device_id.device_uuid.uuid,
                    'device_name': device.name,
                    'device_type': device.device_type,
                    'status': device.device_operational_status,
                    'drivers': [driver for driver in device.device_drivers],
                    'endpoints': [{
                        'uuid': endpoint.endpoint_id.endpoint_uuid.uuid,
                        'name': endpoint.name,
                        'type': endpoint.endpoint_type,
                        'location': endpoint.endpoint_location
                    } for endpoint in device.device_endpoints],
                    'configurations': [{
                        'key': config.custom.resource_key,
                        'value': config.custom.resource_value
                    } for config in device.device_config.config_rules],
                    'interfaces': [{
                        'id': interface.qkdi_id,
                        'enabled': interface.enabled,
                        'name': interface.name,
                        'att_point': interface.qkdi_att_point,
                        'capabilities': interface.qkdi_capabilities
                    } for interface in device.qkd_interfaces.qkd_interface],
                    'applications': [{
                        'app_id': app.app_id,
                        'app_qos': app.app_qos,
                        'app_statistics': app.app_statistics,
                        'backing_qkdl_id': app.backing_qkdl_id,
                        'client_app_id': app.client_app_id
                    } for app in device.qkd_applications.qkd_app]
                }
                topology_info['devices'].append(device_info)
            context_info['topologies'].append(topology_info)
        contexts_info.append(context_info)
    
    return contexts_info

def get_detailed_device_info():
    context_info = get_context_topology_info()
    detailed_info = []
    for context in context_info:
        if context['context_name'] == 'admin':
            for topology in context['topologies']:
                if topology['topology_name'] == 'admin':
                    detailed_info.extend(topology['devices'])
    print("Detailed Device Info:", detailed_info)  # Print the detailed device information
    return detailed_info


class TestRetrieveQKDInformation(unittest.TestCase):

    @patch('retrieve_qkd_information.ContextServiceStub')
    def test_get_detailed_device_info(self, MockContextServiceStub):
        # Mocking the gRPC channel and stubs
        mock_stub = MagicMock()
        MockContextServiceStub.return_value = mock_stub

        # Create a mock response for ListContexts
        context_id = ContextId(context_uuid=Uuid(uuid="43813baf-195e-5da6-af20-b3d0922e71a7"))
        context_response = MagicMock()
        mock_context = MagicMock()
        mock_context.context_id = context_id
        mock_context.name = "admin"
        context_response.contexts = [mock_context]

        # Create a mock response for ListTopologies
        topology_id = TopologyId(topology_uuid=Uuid(uuid="c76135e3-24a8-5e92-9bed-c3c9139359c8"))
        topology_response = MagicMock()
        mock_topology = MagicMock()
        mock_topology.topology_id = topology_id
        mock_topology.name = "admin"
        topology_response.topologies = [mock_topology]

        # Create a mock response for GetTopologyDetails
        device1 = MagicMock()
        device1.device_id.device_uuid.uuid = "40e6c9e2-fdc8-5802-8361-413286c03494"
        device1.name = "QKD2"
        device1.device_type = "qkd-node"
        device1.device_operational_status = "ENABLED"
        device1.device_drivers = ["QKD"]
        
        endpoint1_1 = MagicMock()
        endpoint1_1.endpoint_id.endpoint_uuid.uuid = "97b3b8e2-0e3e-5271-bc1e-ab2600b17fbd"
        endpoint1_1.name = "10.211.36.220:2001"
        endpoint1_1.endpoint_type = "-"
        endpoint1_1.endpoint_location = ""

        endpoint1_2 = MagicMock()
        endpoint1_2.endpoint_id.endpoint_uuid.uuid = "bcb1cc4b-9208-54d1-bb70-8039871dd820"
        endpoint1_2.name = "10.211.36.220:2002"
        endpoint1_2.endpoint_type = "-"
        endpoint1_2.endpoint_location = ""

        device1.device_endpoints = [endpoint1_1, endpoint1_2]

        config_rule1_1 = MagicMock()
        config_rule1_1.custom.resource_key = "_connect/address"
        config_rule1_1.custom.resource_value = "10.211.36.220"

        config_rule1_2 = MagicMock()
        config_rule1_2.custom.resource_key = "_connect/port"
        config_rule1_2.custom.resource_value = "22222"

        config_rule1_3 = MagicMock()
        config_rule1_3.custom.resource_key = "_connect/settings"
        config_rule1_3.custom.resource_value = "scheme: http"

        device1.device_config.config_rules = [config_rule1_1, config_rule1_2, config_rule1_3]

        interface1_1 = MagicMock()
        interface1_1.qkdi_id = "200"
        interface1_1.enabled = True
        interface1_1.name = "10.211.36.220:2001"
        interface1_1.qkdi_att_point = {'device': '10.211.36.220', 'port': '2001'}
        interface1_1.qkdi_capabilities = {}

        interface1_2 = MagicMock()
        interface1_2.qkdi_id = "201"
        interface1_2.enabled = True
        interface1_2.name = "10.211.36.220:2002"
        interface1_2.qkdi_att_point = {'device': '10.211.36.220', 'port': '2002'}
        interface1_2.qkdi_capabilities = {}

        device1.qkd_interfaces.qkd_interface = [interface1_1, interface1_2]

        app1_1 = MagicMock()
        app1_1.app_id = "00000002-0001-0000-0000-000000000000"
        app1_1.app_qos = {}
        app1_1.app_statistics = {'statistics': []}
        app1_1.backing_qkdl_id = []
        app1_1.client_app_id = []

        device1.qkd_applications.qkd_app = [app1_1]

        # Repeat similar structure for device2 and device3
        device2 = MagicMock()
        device2.device_id.device_uuid.uuid = "456e461e-1de7-569a-999f-73903e818e4c"
        device2.name = "QKD3"
        device2.device_type = "qkd-node"
        device2.device_operational_status = "ENABLED"
        device2.device_drivers = ["QKD"]

        endpoint2_1 = MagicMock()
        endpoint2_1.endpoint_id.endpoint_uuid.uuid = "73b56f99-52f3-5af9-a7fd-cdd6e94fb289"
        endpoint2_1.name = "10.211.36.220:3001"
        endpoint2_1.endpoint_type = "-"
        endpoint2_1.endpoint_location = ""

        device2.device_endpoints = [endpoint2_1]

        config_rule2_1 = MagicMock()
        config_rule2_1.custom.resource_key = "_connect/address"
        config_rule2_1.custom.resource_value = "10.211.36.220"

        config_rule2_2 = MagicMock()
        config_rule2_2.custom.resource_key = "_connect/port"
        config_rule2_2.custom.resource_value = "33333"

        config_rule2_3 = MagicMock()
        config_rule2_3.custom.resource_key = "_connect/settings"
        config_rule2_3.custom.resource_value = "scheme: http"

        device2.device_config.config_rules = [config_rule2_1, config_rule2_2, config_rule2_3]

        interface2_1 = MagicMock()
        interface2_1.qkdi_id = "300"
        interface2_1.enabled = True
        interface2_1.name = "10.211.36.220:3001"
        interface2_1.qkdi_att_point = {'device': '10.211.36.220', 'port': '3001'}
        interface2_1.qkdi_capabilities = {}

        device2.qkd_interfaces.qkd_interface = [interface2_1]

        app2_1 = MagicMock()
        app2_1.app_id = "00000003-0001-0000-0000-000000000000"
        app2_1.app_qos = {}
        app2_1.app_statistics = {'statistics': []}
        app2_1.backing_qkdl_id = []
        app2_1.client_app_id = []

        device2.qkd_applications.qkd_app = [app2_1]

        device3 = MagicMock()
        device3.device_id.device_uuid.uuid = "74520336-c12f-545e-9e18-15319f987352"
        device3.name = "QKD1"
        device3.device_type = "qkd-node"
        device3.device_operational_status = "ENABLED"
        device3.device_drivers = ["QKD"]

        endpoint3_1 = MagicMock()
        endpoint3_1.endpoint_id.endpoint_uuid.uuid = "197a413f-5051-5241-81b7-ea4f89f0a0fc"
        endpoint3_1.name = "10.211.36.220:1001"
        endpoint3_1.endpoint_type = "-"
        endpoint3_1.endpoint_location = ""

        device3.device_endpoints = [endpoint3_1]

        config_rule3_1 = MagicMock()
        config_rule3_1.custom.resource_key = "_connect/address"
        config_rule3_1.custom.resource_value = "10.211.36.220"

        config_rule3_2 = MagicMock()
        config_rule3_2.custom.resource_key = "_connect/port"
        config_rule3_2.custom.resource_value = "11111"

        config_rule3_3 = MagicMock()
        config_rule3_3.custom.resource_key = "_connect/settings"
        config_rule3_3.custom.resource_value = "scheme: http"

        device3.device_config.config_rules = [config_rule3_1, config_rule3_2, config_rule3_3]

        interface3_1 = MagicMock()
        interface3_1.qkdi_id = "100"
        interface3_1.enabled = True
        interface3_1.name = "10.211.36.220:1001"
        interface3_1.qkdi_att_point = {'device': '10.211.36.220', 'port': '1001'}
        interface3_1.qkdi_capabilities = {}

        device3.qkd_interfaces.qkd_interface = [interface3_1]

        app3_1 = MagicMock()
        app3_1.app_id = "00000001-0001-0000-0000-000000000000"
        app3_1.app_qos = {}
        app3_1.app_statistics = {'statistics': []}
        app3_1.backing_qkdl_id = []
        app3_1.client_app_id = []

        device3.qkd_applications.qkd_app = [app3_1]

        topology_details_response = MagicMock(devices=[device1, device2, device3])

        # Set up the mock return values
        mock_stub.ListContexts.return_value = context_response
        mock_stub.ListTopologies.return_value = topology_response
        mock_stub.GetTopologyDetails.return_value = topology_details_response

        # Run the function to test
        detailed_info = get_detailed_device_info()

        # Print the detailed information for testing purposes
        print("Test Detailed Info:", detailed_info)

        # Assertions
        mock_stub.ListContexts.assert_called_once_with(Empty())
        mock_stub.ListTopologies.assert_called_once_with(context_id)
        mock_stub.GetTopologyDetails.assert_called_once_with(topology_id)

        # Check the returned information
        expected_info = [{
            'device_id': '40e6c9e2-fdc8-5802-8361-413286c03494',
            'device_name': 'QKD2',
            'device_type': 'qkd-node',
            'status': 'ENABLED',
            'drivers': ['QKD'],
            'endpoints': [
                {'uuid': '97b3b8e2-0e3e-5271-bc1e-ab2600b17fbd', 'name': '10.211.36.220:2001', 'type': '-', 'location': ''},
                {'uuid': 'bcb1cc4b-9208-54d1-bb70-8039871dd820', 'name': '10.211.36.220:2002', 'type': '-', 'location': ''}
            ],
            'configurations': [
                {'key': '_connect/address', 'value': '10.211.36.220'},
                {'key': '_connect/port', 'value': '22222'},
                {'key': '_connect/settings', 'value': 'scheme: http'}
            ],
            'interfaces': [
                {'id': '200', 'enabled': True, 'name': '10.211.36.220:2001', 'att_point': {'device': '10.211.36.220', 'port': '2001'}, 'capabilities': {}},
                {'id': '201', 'enabled': True, 'name': '10.211.36.220:2002', 'att_point': {'device': '10.211.36.220', 'port': '2002'}, 'capabilities': {}}
            ],
            'applications': [
                {'app_id': '00000002-0001-0000-0000-000000000000', 'app_qos': {}, 'app_statistics': {'statistics': []}, 'backing_qkdl_id': [], 'client_app_id': []}
            ]
        }, {
            'device_id': '456e461e-1de7-569a-999f-73903e818e4c',
            'device_name': 'QKD3',
            'device_type': 'qkd-node',
            'status': 'ENABLED',
            'drivers': ['QKD'],
            'endpoints': [
                {'uuid': '73b56f99-52f3-5af9-a7fd-cdd6e94fb289', 'name': '10.211.36.220:3001', 'type': '-', 'location': ''}
            ],
            'configurations': [
                {'key': '_connect/address', 'value': '10.211.36.220'},
                {'key': '_connect/port', 'value': '33333'},
                {'key': '_connect/settings', 'value': 'scheme: http'}
            ],
            'interfaces': [
                {'id': '300', 'enabled': True, 'name': '10.211.36.220:3001', 'att_point': {'device': '10.211.36.220', 'port': '3001'}, 'capabilities': {}}
            ],
            'applications': [
                {'app_id': '00000003-0001-0000-0000-000000000000', 'app_qos': {}, 'app_statistics': {'statistics': []}, 'backing_qkdl_id': [], 'client_app_id': []}
            ]
        }, {
            'device_id': '74520336-c12f-545e-9e18-15319f987352',
            'device_name': 'QKD1',
            'device_type': 'qkd-node',
            'status': 'ENABLED',
            'drivers': ['QKD'],
            'endpoints': [
                {'uuid': '197a413f-5051-5241-81b7-ea4f89f0a0fc', 'name': '10.211.36.220:1001', 'type': '-', 'location': ''}
            ],
            'configurations': [
                {'key': '_connect/address', 'value': '10.211.36.220'},
                {'key': '_connect/port', 'value': '11111'},
                {'key': '_connect/settings', 'value': 'scheme: http'}
            ],
            'interfaces': [
                {'id': '100', 'enabled': True, 'name': '10.211.36.220:1001', 'att_point': {'device': '10.211.36.220', 'port': '1001'}, 'capabilities': {}}
            ],
            'applications': [
                {'app_id': '00000001-0001-0000-0000-000000000000', 'app_qos': {}, 'app_statistics': {'statistics': []}, 'backing_qkdl_id': [], 'client_app_id': []}
            ]
        }]
        self.assertEqual(detailed_info, expected_info)

if __name__ == "__main__":
    unittest.main()
