# Copyright 2021-2023 H2020 TeraFlow (https://www.teraflow-h2020.eu/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import json, logging, operator, requests
from requests.auth import HTTPBasicAuth
from typing import Optional
from device.service.driver_api._Driver import RESOURCE_ENDPOINTS

LOGGER = logging.getLogger(__name__)

HTTP_OK_CODES = {
    200,    # OK
    201,    # Created
    202,    # Accepted
    204,    # No Content
}

def find_key(resource, key):
    return json.loads(resource[1])[key]


def config_getter(
    root_url : str, resource_key : str, auth : Optional[HTTPBasicAuth] = None, timeout : Optional[int] = None
):
    url = '{:s}/restconf/data/tapi-common:context'.format(root_url)
    result = []
    try:
        response = requests.get(url, timeout=timeout, verify=False, auth=auth)
    except requests.exceptions.Timeout:
        LOGGER.exception('Timeout connecting {:s}'.format(url))
        return result
    except Exception as e:  # pylint: disable=broad-except
        LOGGER.exception('Exception retrieving {:s}'.format(resource_key))
        result.append((resource_key, e))
        return result

    try:
        context = json.loads(response.content)
    except Exception as e:  # pylint: disable=broad-except
        LOGGER.warning('Unable to decode reply: {:s}'.format(str(response.content)))
        result.append((resource_key, e))
        return result

    if resource_key != RESOURCE_ENDPOINTS: return result

    if 'tapi-common:context' in context:
        context = context['tapi-common:context']
    elif 'context' in context:
        context = context['context']

    for sip in context['service-interface-point']:
        layer_protocol_name = sip.get('layer-protocol-name', '?')
        supportable_spectrum = sip.get('tapi-photonic-media:media-channel-service-interface-point-spec', {})
        supportable_spectrum = supportable_spectrum.get('mc-pool', {})
        supportable_spectrum = supportable_spectrum.get('supportable-spectrum', [])
        supportable_spectrum = supportable_spectrum[0] if len(supportable_spectrum) == 1 else {}
        grid_type = supportable_spectrum.get('frequency-constraint', {}).get('grid-type')
        granularity = supportable_spectrum.get('frequency-constraint', {}).get('adjustment-granularity')
        direction = sip.get('direction', '?')
        endpoint_type = [layer_protocol_name, grid_type, granularity, direction]
        str_endpoint_type = ':'.join(filter(lambda i: operator.is_not(i, None), endpoint_type))
        endpoint_url = '/endpoints/endpoint[{:s}]'.format(sip['uuid'])
        endpoint_data = {'uuid': sip['uuid'], 'type': str_endpoint_type}
        result.append((endpoint_url, endpoint_data))

    return result

def create_connectivity_service(
    root_url, uuid, input_sip, output_sip, direction, capacity_value, capacity_unit, layer_protocol_name,
    layer_protocol_qualifier,
    auth : Optional[HTTPBasicAuth] = None, timeout : Optional[int] = None
):

    url = '{:s}/restconf/data/tapi-common:context/tapi-connectivity:connectivity-context'.format(root_url)
    headers = {'content-type': 'application/json'}
    data = {
        'tapi-connectivity:connectivity-service': [
            {
                'uuid': uuid,
                'connectivity-constraint': {
                    'requested-capacity': {
                        'total-size': {
                            'value': capacity_value,
                            'unit': capacity_unit
                        }
                    },
                    'connectivity-direction': direction
                },
                'end-point': [
                    {
                        'service-interface-point': {
                            'service-interface-point-uuid': input_sip
                        },
                        'layer-protocol-name': layer_protocol_name,
                        'layer-protocol-qualifier': layer_protocol_qualifier,
                        'local-id': input_sip
                    },
                    {
                        'service-interface-point': {
                            'service-interface-point-uuid': output_sip
                        },
                        'layer-protocol-name': layer_protocol_name,
                        'layer-protocol-qualifier': layer_protocol_qualifier,
                        'local-id': output_sip
                    }
                ]
            }
        ]
    }
    results = []
    try:
        LOGGER.info('Connectivity service {:s}: {:s}'.format(str(uuid), str(data)))
        response = requests.post(
            url=url, data=json.dumps(data), timeout=timeout, headers=headers, verify=False, auth=auth)
        LOGGER.info('TAPI response: {:s}'.format(str(response)))
    except Exception as e:  # pylint: disable=broad-except
        LOGGER.exception('Exception creating ConnectivityService(uuid={:s}, data={:s})'.format(str(uuid), str(data)))
        results.append(e)
    else:
        if response.status_code not in HTTP_OK_CODES:
            msg = 'Could not create ConnectivityService(uuid={:s}, data={:s}). status_code={:s} reply={:s}'
            LOGGER.error(msg.format(str(uuid), str(data), str(response.status_code), str(response)))
        results.append(response.status_code in HTTP_OK_CODES)
    return results

def delete_connectivity_service(root_url, uuid, auth : Optional[HTTPBasicAuth] = None, timeout : Optional[int] = None):
    url = '{:s}/restconf/data/tapi-common:context/tapi-connectivity:connectivity-context/connectivity-service={:s}'
    url = url.format(root_url, uuid)
    results = []
    try:
        response = requests.delete(url=url, timeout=timeout, verify=False, auth=auth)
    except Exception as e:  # pylint: disable=broad-except
        LOGGER.exception('Exception deleting ConnectivityService(uuid={:s})'.format(str(uuid)))
        results.append(e)
    else:
        if response.status_code not in HTTP_OK_CODES:
            msg = 'Could not delete ConnectivityService(uuid={:s}). status_code={:s} reply={:s}'
            LOGGER.error(msg.format(str(uuid), str(response.status_code), str(response)))
        results.append(response.status_code in HTTP_OK_CODES)
    return results
