# Copyright 2021-2023 H2020 TeraFlow (https://www.teraflow-h2020.eu/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from sqlalchemy.dialects.postgresql import insert
from sqlalchemy.engine import Engine
from sqlalchemy.orm import Session, sessionmaker
from sqlalchemy_cockroachdb import run_transaction
from typing import Dict, List, Optional, Set, Tuple
from common.proto.context_pb2 import Device, DeviceId, DeviceIdList, DeviceList
from common.method_wrappers.ServiceExceptions import InvalidArgumentException, NotFoundException
from common.tools.object_factory.Device import json_device_id
from context.service.database.ConfigRule import compose_config_rules_data, upsert_config_rules
from .models.DeviceModel import DeviceModel
from .models.EndPointModel import EndPointModel
from .models.TopologyModel import TopologyDeviceModel
from .models.enums.DeviceDriver import grpc_to_enum__device_driver
from .models.enums.DeviceOperationalStatus import grpc_to_enum__device_operational_status
from .models.enums.KpiSampleType import grpc_to_enum__kpi_sample_type
from .uuids.Device import device_get_uuid
from .uuids.EndPoint import endpoint_get_uuid

def device_list_ids(db_engine : Engine) -> DeviceIdList:
    def callback(session : Session) -> List[Dict]:
        obj_list : List[DeviceModel] = session.query(DeviceModel).all()
        #.options(selectinload(DeviceModel.topology)).filter_by(context_uuid=context_uuid).one_or_none()
        return [obj.dump_id() for obj in obj_list]
    return DeviceIdList(device_ids=run_transaction(sessionmaker(bind=db_engine), callback))

def device_list_objs(db_engine : Engine) -> DeviceList:
    def callback(session : Session) -> List[Dict]:
        obj_list : List[DeviceModel] = session.query(DeviceModel).all()
        #.options(selectinload(DeviceModel.topology)).filter_by(context_uuid=context_uuid).one_or_none()
        return [obj.dump() for obj in obj_list]
    return DeviceList(devices=run_transaction(sessionmaker(bind=db_engine), callback))

def device_get(db_engine : Engine, request : DeviceId) -> Device:
    device_uuid = device_get_uuid(request, allow_random=False)
    def callback(session : Session) -> Optional[Dict]:
        obj : Optional[DeviceModel] = session.query(DeviceModel).filter_by(device_uuid=device_uuid).one_or_none()
        return None if obj is None else obj.dump()
    obj = run_transaction(sessionmaker(bind=db_engine), callback)
    if obj is None:
        raw_device_uuid = request.device_uuid.uuid
        raise NotFoundException('Device', raw_device_uuid, extra_details=[
            'device_uuid generated was: {:s}'.format(device_uuid)
        ])
    return Device(**obj)

def device_set(db_engine : Engine, request : Device) -> Tuple[DeviceId, bool]:
    raw_device_uuid = request.device_id.device_uuid.uuid
    raw_device_name = request.name
    device_name = raw_device_uuid if len(raw_device_name) == 0 else raw_device_name
    device_uuid = device_get_uuid(request.device_id, device_name=device_name, allow_random=True)

    device_type = request.device_type
    oper_status = grpc_to_enum__device_operational_status(request.device_operational_status)
    device_drivers = [grpc_to_enum__device_driver(d) for d in request.device_drivers]

    topology_uuids : Set[str] = set()
    related_topologies : List[Dict] = list()
    endpoints_data : List[Dict] = list()
    for i, endpoint in enumerate(request.device_endpoints):
        endpoint_device_uuid = endpoint.endpoint_id.device_id.device_uuid.uuid
        if len(endpoint_device_uuid) == 0: endpoint_device_uuid = device_uuid
        if endpoint_device_uuid not in {raw_device_uuid, device_uuid}:
            raise InvalidArgumentException(
                'request.device_endpoints[{:d}].device_id.device_uuid.uuid'.format(i), endpoint_device_uuid,
                ['should be == request.device_id.device_uuid.uuid({:s})'.format(raw_device_uuid)]
            )

        raw_endpoint_uuid = endpoint.endpoint_id.endpoint_uuid.uuid
        raw_endpoint_name = endpoint.name
        endpoint_topology_uuid, endpoint_device_uuid, endpoint_uuid = endpoint_get_uuid(
            endpoint.endpoint_id, endpoint_name=raw_endpoint_name, allow_random=True)
        endpoint_name = raw_endpoint_uuid if len(raw_endpoint_name) == 0 else raw_endpoint_name

        kpi_sample_types = [grpc_to_enum__kpi_sample_type(kst) for kst in endpoint.kpi_sample_types]

        endpoints_data.append({
            'endpoint_uuid'   : endpoint_uuid,
            'device_uuid'     : endpoint_device_uuid,
            'topology_uuid'   : endpoint_topology_uuid,
            'name'            : endpoint_name,
            'endpoint_type'   : endpoint.endpoint_type,
            'kpi_sample_types': kpi_sample_types,
        })

        if endpoint_topology_uuid not in topology_uuids:
            related_topologies.append({
                'topology_uuid': endpoint_topology_uuid,
                'device_uuid'  : device_uuid,
            })
            topology_uuids.add(endpoint_topology_uuid)

    config_rules = compose_config_rules_data(request.device_config.config_rules, device_uuid=device_uuid)

    device_data = [{
        'device_uuid'              : device_uuid,
        'device_name'              : device_name,
        'device_type'              : device_type,
        'device_operational_status': oper_status,
        'device_drivers'           : device_drivers,
    }]

    def callback(session : Session) -> None:
        stmt = insert(DeviceModel).values(device_data)
        stmt = stmt.on_conflict_do_update(
            index_elements=[DeviceModel.device_uuid],
            set_=dict(
                device_name               = stmt.excluded.device_name,
                device_type               = stmt.excluded.device_type,
                device_operational_status = stmt.excluded.device_operational_status,
                device_drivers            = stmt.excluded.device_drivers,
            )
        )
        session.execute(stmt)

        stmt = insert(EndPointModel).values(endpoints_data)
        stmt = stmt.on_conflict_do_update(
            index_elements=[EndPointModel.endpoint_uuid],
            set_=dict(
                name             = stmt.excluded.name,
                endpoint_type    = stmt.excluded.endpoint_type,
                kpi_sample_types = stmt.excluded.kpi_sample_types,
            )
        )
        session.execute(stmt)

        session.execute(insert(TopologyDeviceModel).values(related_topologies).on_conflict_do_nothing(
            index_elements=[TopologyDeviceModel.topology_uuid, TopologyDeviceModel.device_uuid]
        ))

        upsert_config_rules(session, config_rules, device_uuid=device_uuid)

    run_transaction(sessionmaker(bind=db_engine), callback)
    updated = False # TODO: improve and check if created/updated
    return DeviceId(**json_device_id(device_uuid)),updated

def device_delete(db_engine : Engine, request : DeviceId) -> bool:
    device_uuid = device_get_uuid(request, allow_random=False)
    def callback(session : Session) -> bool:
        num_deleted = session.query(DeviceModel).filter_by(device_uuid=device_uuid).delete()
        return num_deleted > 0
    return run_transaction(sessionmaker(bind=db_engine), callback)
