# Copyright 2021-2023 H2020 TeraFlow (https://www.teraflow-h2020.eu/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import logging, threading
from common.tools.grpc.Tools import grpc_message_to_json_string
from context.client.ContextClient import ContextClient
from context.client.EventsCollector import EventsCollector
from dlt.connector.client.DltClient import DltClient

LOGGER = logging.getLogger(__name__)

class DltConnector:
    def __init__(self) -> None:
        LOGGER.debug('Creating connector...')
        self._terminate = threading.Event()
        self._thread = None
        LOGGER.debug('Connector created')

    def start(self):
        self._terminate.clear()
        self._thread = threading.Thread(target=self._run_events_collector)
        self._thread.start()

    def _run_events_collector(self) -> None:
        dlt_client = DltClient()
        context_client = ContextClient()
        events_collector = EventsCollector(context_client)
        events_collector.start()

        while not self._terminate.is_set():
            event = events_collector.get_event()
            LOGGER.info('Event from Context Received: {:s}'.format(grpc_message_to_json_string(event)))

        events_collector.stop()
        context_client.close()
        dlt_client.close()

    def stop(self):
        self._terminate.set()
        self._thread.join()
