# Copyright 2022-2024 ETSI SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import copy
import threading
from typing import Dict, Optional, Tuple
from common.proto.kpi_sample_types_pb2 import KpiSampleType

class ResourceKeyMapper:
    def __init__(self) -> None:
        self._lock_device_endpoint = threading.Lock()
        self._device_endpoint_sampletype__to__resource_key : Dict[Tuple[str, str, int], str] = dict()

    def add_resource_key(
        self, device_uuid : str, endpoint_uuid : str, kpi_sample_type : KpiSampleType, resource_key : str
    ) -> None:
        with self._lock_device_endpoint:
            key = (device_uuid, endpoint_uuid, kpi_sample_type)
            self._device_endpoint_sampletype__to__resource_key[key] = resource_key

    def get_resource_key(
        self, device_uuid : str, endpoint_uuid : str, kpi_sample_type : KpiSampleType
    ) -> Optional[str]:
        with self._lock_device_endpoint:
            key = (device_uuid, endpoint_uuid, kpi_sample_type)
            return self._device_endpoint_sampletype__to__resource_key.get(key)

    def get_all_resource_keys(self) -> Dict[Tuple[str, str, int], str]:
        with self._lock_device_endpoint:
            return copy.deepcopy(self._device_endpoint_sampletype__to__resource_key)

    def remove_resource_key(
        self, device_uuid : str, endpoint_uuid : str, kpi_sample_type : KpiSampleType
    ) -> None:
        with self._lock_device_endpoint:
            key = (device_uuid, endpoint_uuid, kpi_sample_type)
            self._device_endpoint_sampletype__to__resource_key.pop(key, None)

