# Copyright 2022-2023 ETSI TeraFlowSDN - TFS OSG (https://tfs.etsi.org/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import json
from flask.json import jsonify
from flask_restful import Resource, request
from common.proto.context_pb2 import Empty
from common.tools.grpc.Tools import grpc_message_to_json
from context.client.ContextClient import ContextClient
from service.client.ServiceClient import ServiceClient
from vnt_manager.client.VNTManagerClient import VNTManagerClient

from .Tools import (
    format_grpc_to_json, grpc_connection_id, grpc_context_id, grpc_device_id, grpc_link_id, grpc_policy_rule_id,
    grpc_service_id, grpc_service, grpc_slice_id, grpc_topology_id)

class _Resource(Resource):
    def __init__(self) -> None:
        super().__init__()
        self.client = ContextClient()
        self.service_client = ServiceClient()
        self.vntmanager_client = VNTManagerClient()

class ContextIds(_Resource):
    def get(self):
        return format_grpc_to_json(self.client.ListContextIds(Empty()))

class Contexts(_Resource):
    def get(self):
        return format_grpc_to_json(self.client.ListContexts(Empty()))

class DummyContexts(_Resource):
    def get(self):
        contexts = grpc_message_to_json(self.client.ListContexts(Empty()), use_integers_for_enums=True)['contexts']
        devices = grpc_message_to_json(self.client.ListDevices(Empty()), use_integers_for_enums=True)['devices']
        links = grpc_message_to_json(self.client.ListLinks(Empty()), use_integers_for_enums=True)['links']

        topologies  = list()
        slices      = list()
        services    = list()
        connections = list()

        for context in contexts:
            context_uuid = context['context_id']['context_uuid']['uuid']
            context_id = grpc_context_id(context_uuid)

            topologies.extend(grpc_message_to_json(
                self.client.ListTopologies(context_id),
                use_integers_for_enums=True
            )['topologies'])

            slices.extend(grpc_message_to_json(
                self.client.ListSlices(context_id),
                use_integers_for_enums=True
            )['slices'])

            context_services = grpc_message_to_json(
                self.client.ListServices(context_id),
                use_integers_for_enums=True
            )['services']
            services.extend(context_services)

            for service in context_services:
                service_uuid = service['service_id']['service_uuid']['uuid']
                service_id = grpc_service_id(context_uuid, service_uuid)
                connections.extend(grpc_message_to_json(
                    self.client.ListConnections(service_id),
                    use_integers_for_enums=True
                )['connections'])

        for device in devices:
            for config_rule in device['device_config']['config_rules']:
                if 'custom' not in config_rule: continue
                resource_value = config_rule['custom']['resource_value']
                if not isinstance(resource_value, str): continue
                try:
                    resource_value = json.loads(resource_value)
                except: # pylint: disable=bare-except
                    pass
                config_rule['custom']['resource_value'] = resource_value

        dummy_context = {'dummy_mode': True}
        if len(contexts   ) > 0: dummy_context['contexts'   ] = contexts
        if len(topologies ) > 0: dummy_context['topologies' ] = topologies
        if len(devices    ) > 0: dummy_context['devices'    ] = devices
        if len(links      ) > 0: dummy_context['links'      ] = links
        if len(slices     ) > 0: dummy_context['slices'     ] = slices
        if len(services   ) > 0: dummy_context['services'   ] = services
        if len(connections) > 0: dummy_context['connections'] = connections
        return jsonify(dummy_context)

class Context(_Resource):
    def get(self, context_uuid : str):
        return format_grpc_to_json(self.client.GetContext(grpc_context_id(context_uuid)))

class TopologyIds(_Resource):
    def get(self, context_uuid : str):
        return format_grpc_to_json(self.client.ListTopologyIds(grpc_context_id(context_uuid)))

class Topologies(_Resource):
    def get(self, context_uuid : str):
        return format_grpc_to_json(self.client.ListTopologies(grpc_context_id(context_uuid)))

class Topology(_Resource):
    def get(self, context_uuid : str, topology_uuid : str):
        return format_grpc_to_json(self.client.GetTopology(grpc_topology_id(context_uuid, topology_uuid)))

class ServiceIds(_Resource):
    def get(self, context_uuid : str):
        return format_grpc_to_json(self.client.ListServiceIds(grpc_context_id(context_uuid)))

class Services(_Resource):
    def get(self, context_uuid : str):
        return format_grpc_to_json(self.client.ListServices(grpc_context_id(context_uuid)))

class Service(_Resource):
    def get(self, context_uuid : str, service_uuid : str):
        return format_grpc_to_json(self.client.GetService(grpc_service_id(context_uuid, service_uuid)))

    def post(self, context_uuid : str, service_uuid : str): # pylint: disable=unused-argument
        service = request.get_json()['services'][0]
        return format_grpc_to_json(self.service_client.CreateService(grpc_service(
            service_uuid = service['service_id']['service_uuid']['uuid'],
            service_type = service['service_type'],
            context_uuid = service['service_id']['context_id']['context_uuid']['uuid'],
        )))

    def put(self, context_uuid : str, service_uuid : str):  # pylint: disable=unused-argument
        service = request.get_json()['services'][0]
        return format_grpc_to_json(self.service_client.UpdateService(grpc_service(
            service_uuid = service['service_id']['service_uuid']['uuid'],
            service_type = service['service_type'],
            context_uuid = service['service_id']['context_id']['context_uuid']['uuid'],
            status       = service['service_status']['service_status'],
            endpoint_ids = service['service_endpoint_ids'],
            constraints  = service['service_constraints'],
            config_rules = service['service_config']['config_rules']
        )))

    def delete(self, context_uuid : str, service_uuid : str):
        return format_grpc_to_json(self.service_client.DeleteService(grpc_service_id(
            context_uuid, service_uuid,
        )))

class SliceIds(_Resource):
    def get(self, context_uuid : str):
        return format_grpc_to_json(self.client.ListSliceIds(grpc_context_id(context_uuid)))

class Slices(_Resource):
    def get(self, context_uuid : str):
        return format_grpc_to_json(self.client.ListSlices(grpc_context_id(context_uuid)))

class Slice(_Resource):
    def get(self, context_uuid : str, slice_uuid : str):
        return format_grpc_to_json(self.client.GetSlice(grpc_slice_id(context_uuid, slice_uuid)))

class DeviceIds(_Resource):
    def get(self):
        return format_grpc_to_json(self.client.ListDeviceIds(Empty()))

class Devices(_Resource):
    def get(self):
        return format_grpc_to_json(self.client.ListDevices(Empty()))

class Device(_Resource):
    def get(self, device_uuid : str):
        return format_grpc_to_json(self.client.GetDevice(grpc_device_id(device_uuid)))

class LinkIds(_Resource):
    def get(self):
        return format_grpc_to_json(self.client.ListLinkIds(Empty()))

class Links(_Resource):
    def get(self):
        return format_grpc_to_json(self.client.ListLinks(Empty()))

class Link(_Resource):
    def get(self, link_uuid : str):
        return format_grpc_to_json(self.client.GetLink(grpc_link_id(link_uuid)))

class VirtualLinkIds(_Resource):
    def get(self):
        return format_grpc_to_json(self.vntmanager_client.ListLinkIds(Empty()))

class VirtualLinks(_Resource):
    def get(self):
        return format_grpc_to_json(self.vntmanager_client.ListLinks(Empty()))

class VirtualLink(_Resource):
    def get(self, link_uuid : str):
        return format_grpc_to_json(self.vntmanager_client.GetLink(grpc_link_id(link_uuid)))
    def post(self, link_uuid : str):
        link = request.get_json()
        return format_grpc_to_json(self.vntmanager_client.SetLink(grpc_link_id(link)))
    def put(self, link_uuid : str):
        link = request.get_json()
        return format_grpc_to_json(self.vntmanager_client.SetLink(grpc_link_id(link)))
    def delete(self, link_uuid : str):
        return format_grpc_to_json(self.vntmanager_client.RemoveVirtualLink(grpc_link_id(link_uuid)))

class ConnectionIds(_Resource):
    def get(self, context_uuid : str, service_uuid : str):
        return format_grpc_to_json(self.client.ListConnectionIds(grpc_service_id(context_uuid, service_uuid)))

class Connections(_Resource):
    def get(self, context_uuid : str, service_uuid : str):
        return format_grpc_to_json(self.client.ListConnections(grpc_service_id(context_uuid, service_uuid)))

class Connection(_Resource):
    def get(self, connection_uuid : str):
        return format_grpc_to_json(self.client.GetConnection(grpc_connection_id(connection_uuid)))

class PolicyRuleIds(_Resource):
    def get(self):
        return format_grpc_to_json(self.client.ListPolicyRuleIds(Empty()))

class PolicyRules(_Resource):
    def get(self):
        return format_grpc_to_json(self.client.ListPolicyRules(Empty()))

class PolicyRule(_Resource):
    def get(self, policy_rule_uuid : str):
        return format_grpc_to_json(self.client.GetPolicyRule(grpc_policy_rule_id(policy_rule_uuid)))
