# Copyright 2022-2023 ETSI TeraFlowSDN - TFS OSG (https://tfs.etsi.org/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import json
import logging
import re
import time
from decimal import ROUND_HALF_EVEN, Decimal
from flask.json import jsonify
from common.proto.context_pb2 import (
    ContextId, Empty, EndPointId, ServiceId, ServiceTypeEnum, Service, ServiceStatusEnum, Constraint, Constraint_SLA_Capacity,
    ConfigRule, ConfigRule_Custom, ConfigActionEnum)
from common.tools.grpc.Tools import grpc_message_to_json
from common.tools.grpc.ConfigRules import update_config_rule_custom
from common.tools.object_factory.Context import json_context_id
from common.tools.object_factory.Service import json_service_id

LOGGER = logging.getLogger(__name__)

ENDPOINT_SETTINGS_KEY = '/device[{:s}]/endpoint[{:s}]/vlan[{:d}]/settings'
RE_CONFIG_RULE_IF_SUBIF = re.compile(r'^\/interface\[([^\]]+)\]\/subinterface\[([^\]]+)\]$')
MEC_CONSIDERED_FIELDS = ['requestType', 'sessionFilter', 'fixedAllocation', 'allocationDirection']
ALLOCATION_DIRECTION_DESCRIPTIONS = {
    '00' : 'Downlink (towards the UE)',
    '01' : 'Uplink (towards the application/session)',
    '10' : 'Symmetrical'}
VLAN_TAG = 0
PREFIX_LENGTH = 24
BGP_AS = 65000
policy_AZ = 'srv_{:d}_a'.format(VLAN_TAG)
policy_ZA = 'srv_{:d}_b'.format(VLAN_TAG)

def service_2_bwInfo(service: Service) -> dict:
    response = {}
    response['appInsId'] = service.service_id.service_uuid.uuid # String: Application instance identifier
    for constraint in service.service_constraints:
        if constraint.WhichOneof('constraint') == 'sla_capacity':
            # String: Size of requested fixed BW allocation in [bps]
            fixed_allocation = Decimal(constraint.sla_capacity.capacity_gbps * 1.e9)
            fixed_allocation = fixed_allocation.quantize(Decimal('0.1'), rounding=ROUND_HALF_EVEN)
            response['fixedAllocation'] = str(fixed_allocation)
            break

    for config_rule in service.service_config.config_rules:
        for key in ['allocationDirection', 'fixedBWPriority', 'requestType', 'sourceIp', 'sourcePort', 'dstPort', 'protocol', 'sessionFilter']:
            if config_rule.custom.resource_key == key:
                if key != 'sessionFilter':
                    response[key] = config_rule.custom.resource_value
                else:
                    response[key] = json.loads(config_rule.custom.resource_value)

    unixtime = time.time()
    response['timeStamp'] = { # Time stamp to indicate when the corresponding information elements are sent
        "seconds": int(unixtime),
        "nanoseconds": int(unixtime%1*1e9)
    }

    return response

def bwInfo_2_service(client, bw_info: dict) -> Service:
    # add description to allocationDirection code
    if ad_code := bw_info.get('allocationDirection'):
        bw_info['allocationDirection'] = {'code': ad_code, 'description': ALLOCATION_DIRECTION_DESCRIPTIONS[ad_code]}
    if 'sessionFilter' in bw_info:
        bw_info['sessionFilter'] = bw_info['sessionFilter'][0] # Discard other items in sessionFilter field

    service = Service()
    
    service_config_rules = service.service_config.config_rules

    route_distinguisher = '{:5d}:{:03d}'.format(BGP_AS, VLAN_TAG)
    settings_cr_key = '/settings'
    settings_cr_value = {'bgp_as':(BGP_AS, True), 'route_distinguisher': (route_distinguisher, True)}
    update_config_rule_custom(service_config_rules, settings_cr_key, settings_cr_value)

    request_cr_key = '/request'
    request_cr_value = {k:bw_info[k] for k in MEC_CONSIDERED_FIELDS}

    config_rule = ConfigRule()
    config_rule.action = ConfigActionEnum.CONFIGACTION_SET
    config_rule_custom = ConfigRule_Custom()
    config_rule_custom.resource_key  = request_cr_key
    config_rule_custom.resource_value = json.dumps(request_cr_value)
    config_rule.custom.CopyFrom(config_rule_custom)
    service_config_rules.append(config_rule)

    if 'sessionFilter' in bw_info:
        a_ip = bw_info['sessionFilter']['sourceIp']
        z_ip = bw_info['sessionFilter']['dstAddress']

        devices = client.ListDevices(Empty()).devices
        router_id_counter = 1
        for device in devices:
            device_endpoint_uuids = {ep.name:ep.endpoint_id.endpoint_uuid.uuid for ep in device.device_endpoints}
            for cr in device.device_config.config_rules:
                if cr.WhichOneof('config_rule') != 'custom':
                    continue
                match_subif = RE_CONFIG_RULE_IF_SUBIF.match(cr.custom.resource_key)
                if not match_subif:
                    continue
                address_ip = json.loads(cr.custom.resource_value).get('address_ip')
                if address_ip not in [a_ip, z_ip]:
                    continue
                port_name = 'PORT-' + match_subif.groups(0)[0] # `PORT-` added as prefix
                ep_id = EndPointId()
                ep_id.endpoint_uuid.uuid = device_endpoint_uuids[port_name]
                ep_id.device_id.device_uuid.uuid = device.device_id.device_uuid.uuid
                service.service_endpoint_ids.append(ep_id)

                # add interface config rules
                endpoint_settings_key = ENDPOINT_SETTINGS_KEY.format(device.name, port_name, VLAN_TAG)
                if address_ip == a_ip:
                    field_updates = {
                    'address_ip': (address_ip, True),
                    # 'router_id': ('.'.join([str(router_id_counter)]*4), True),
                    'router_id': ('200.1.1.1', True),
                    'neighbor_address_ip': ('192.168.150.2', True),
                    'route_distinguisher': (route_distinguisher, True),
                    'sub_interface_index': (0, True),
                    'vlan_id'            : (VLAN_TAG, True),
                    # 'bgp_as': (BGP_AS+router_id_counter, True),
                    'bgp_as': (BGP_AS, True),
                    'ip_address': (address_ip, True),
                    'prefix_length': (PREFIX_LENGTH, True),
                    'policy_AZ'           : (policy_AZ, True),
                    'policy_ZA'           : (policy_ZA, True),
                    'address_prefix'     : (PREFIX_LENGTH, True),
                        }
                elif address_ip == z_ip:
                    field_updates = {
                    'address_ip': (address_ip, True),
                    # 'router_id': ('.'.join([str(router_id_counter)]*4), True),
                    'router_id': ('200.1.1.2', True),
                    'neighbor_address_ip': ('192.168.150.1', True),
                    'route_distinguisher': (route_distinguisher, True),
                    'sub_interface_index': (0, True),
                    'vlan_id'            : (VLAN_TAG, True),
                    # 'bgp_as': (BGP_AS+router_id_counter, True),
                    'bgp_as': (BGP_AS, True),
                    'ip_address': (address_ip, True),
                    'prefix_length': (PREFIX_LENGTH, True),
                    'policy_AZ'           : (policy_ZA, True),
                    'policy_ZA'           : (policy_AZ, True),
                    'address_prefix'     : (PREFIX_LENGTH, True),
                        }
                router_id_counter += 1
                LOGGER.debug(f'BEFORE UPDATE -> device.device_config.config_rules: {service_config_rules}')
                update_config_rule_custom(service_config_rules, endpoint_settings_key, field_updates)
                LOGGER.debug(f'AFTER UPDATE -> device.device_config.config_rules: {service_config_rules}')

    service.service_status.service_status = ServiceStatusEnum.SERVICESTATUS_PLANNED
    service.service_type = ServiceTypeEnum.SERVICETYPE_L3NM

    if 'appInsId' in bw_info:
        service.service_id.service_uuid.uuid = bw_info['appInsId']
        service.service_id.context_id.context_uuid.uuid = 'admin'
        service.name = bw_info['appInsId']

    if 'fixedAllocation' in bw_info:
        capacity = Constraint_SLA_Capacity()
        capacity.capacity_gbps = float(bw_info['fixedAllocation']) / 1.e9
        constraint = Constraint()
        constraint.sla_capacity.CopyFrom(capacity)
        service.service_constraints.append(constraint)

    return service


def format_grpc_to_json(grpc_reply):
    return jsonify(grpc_message_to_json(grpc_reply))

def grpc_context_id(context_uuid):
    return ContextId(**json_context_id(context_uuid))

def grpc_service_id(context_uuid, service_uuid):
    return ServiceId(**json_service_id(service_uuid, context_id=json_context_id(context_uuid)))
