#!/bin/bash
# Copyright 2021-2023 H2020 TeraFlow (https://www.teraflow-h2020.eu/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# If not already set, set the list of components you want to install dependencies for.
# By default, dependencies for all components are installed.
# Components still not supported by this script:
#   automation & policy : implemented in Java
#   dlt                 : under design
#   pathcomp            : under design
ALL_COMPONENTS="context device service compute monitoring webui interdomain slice"
ALL_COMPONENTS="${ALL_COMPONENTS} dbscanserving opticalattackmitigator opticalcentralizedattackdetector"
ALL_COMPONENTS="${ALL_COMPONENTS} l3_attackmitigator l3_centralizedattackdetector l3_distributedattackdetector"
TFS_COMPONENTS=${TFS_COMPONENTS:-$ALL_COMPONENTS}

echo "Updating PIP, SetupTools and Wheel..."
pip install --upgrade pip               # ensure next packages get the latest versions
pip install --upgrade setuptools wheel  # bring basic tooling for other requirements
pip install --upgrade pip-tools pylint  # bring tooling for package compilation and code linting
printf "\n"

echo "Creating integrated requirements file..."
touch requirements.in
diff requirements.in common_requirements.in | grep '^>' | sed 's/^>\ //' >> requirements.in
printf "\n"

echo "Collecting requirements from components..."
for COMPONENT in $TFS_COMPONENTS
do
    if [ "$COMPONENT" == "automation" ] || [ "$COMPONENT" == "policy" ]; then continue; fi
    diff requirements.in src/$COMPONENT/requirements.in | grep '^>' | sed 's/^>\ //' >> requirements.in
    echo "Dependencies from $COMPONENT:"
    for d in `diff requirements.in src/$COMPONENT/requirements.in | grep '^>' | sed 's/^>\ //'`
    do
        echo "    $d"
    done
done
printf "\n"

echo "Compiling requirements..."
# Done in a single step to prevent breaking dependencies between components
pip-compile --quiet --output-file=requirements.txt requirements.in
printf "\n"

echo "Installing requirements..."
python -m pip install -r requirements.txt
printf "\n"

#echo "Removing the temporary files..."
rm requirements.in
rm requirements.txt
printf "\n"
